import { Body, Controller, Delete, Get, HttpStatus, Inject, Post } from '@nestjs/common';
import {
  ApiBody,
  ApiConflictResponse,
  ApiOkResponse,
  ApiTags,
} from '@nestjs/swagger';
import { ApiErrorResponse, PrimeException, PrimeLogger } from '../../definition';
import { UserRoleService, UserService } from 'src/framework/application';
import { LoginOutTO, RecordOutTO, RecordTO, User } from 'src/framework/domain';
import { DecodedToken } from '../../decorators/decoded-token/decoded-token.decorator';
import { DecodedIdToken } from 'firebase-admin/lib/auth/token-verifier';
import { AccessToken } from '../../decorators/access-token/access-token.decorator';
import { inspect } from 'util';
import { ApiDefaultResponses } from '../../decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from '../../decorators/prime-user/prime-user.decorator';
import { ConfigService } from '@nestjs/config';

@Controller('access')
@ApiTags('access')
export class AccessController {
  private readonly LOGGER = new PrimeLogger(AccessController.name);
  private readonly latestVersionAndroid: string;
  private readonly latestVersionIOS: string;
  constructor(
    @Inject('UserService') private readonly userService: UserService,
    @Inject('UserRoleService') private readonly userRoleService: UserRoleService,
      private readonly configService: ConfigService,) {
      this.latestVersionAndroid = this.configService.get<string>(
        'LATEST_VERSION_ANDROID',
        '1.0.0+11'
      );
      this.latestVersionIOS = this.configService.get<string>(
        'LATEST_VERSION_IOS',
        '1.0.0+11'
      );
    }

  
  @Delete('errase-account')
  @ApiDefaultResponses()
  @ApiConflictResponse({ type: ApiErrorResponse })
  @ApiOkResponse({ type: String })
  async erraseAccount(@PrimeUser() user: User): Promise<String> {
    this.LOGGER.log(`erraseAccount user ${user.id}`);
    await this.userService.errase(user);
    this.LOGGER.log(`erraseAccount user ${user.id} deleted`);
    return 'true';
  }
  
  @Post('record-user')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: RecordOutTO })
  @ApiConflictResponse({ type: ApiErrorResponse })
  @ApiBody({ type: RecordTO })
  async recordUserForm(
    @DecodedToken() decodedToken: DecodedIdToken,
    @Body() recordTO: RecordTO,
  ): Promise<RecordOutTO> {
    this.LOGGER.log(`recordUserForm called with decodedToken: ${inspect(decodedToken)}, recordTO: ${inspect(recordTO)}`);
    recordTO.user.externalId = decodedToken.sub;
    return {
      result: await this.userService.registerUserByProfile(recordTO),
    };
  }

  @Get('check-status')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: LoginOutTO })
  @ApiConflictResponse({ type: ApiErrorResponse })
  async checkStatusSession(
    @PrimeUser() user: User,
    @AccessToken() accessToken: string,
  ): Promise<LoginOutTO | undefined> {
    this.LOGGER.log(`checkStatusSession user ${inspect(user)}`);
    const profiles = await this.userRoleService.getRolesByUserId(user.id);
    return {
      fullName: user && `${user.name} ${user.lastName != null ? user.lastName: ''} ${user.lastname2!=null ? user.lastname2: ''}`.trim(),
      token: accessToken,
      cellPhone: user.cellPhone,
      aggrementDate: user.aggrementDate,
      roles: profiles.map(profile => profile).join(', '),
      latestVersionAndroid: this.latestVersionAndroid,
      latestVersionIOS: this.latestVersionIOS,
    };
  }

  @Get('me')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: User })
  async me(@PrimeUser() user: User): Promise<User> {
    this.LOGGER.log(`me called with user: ${inspect(user)}`);
    return user;
  }

  @Get('unsubscribe')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: User })
  async unsubscribe(@PrimeUser() user: User): Promise<String> {
    this.LOGGER.log(`Unsubscribing user ${user.id}`);
    this.userService.unsubscribe(user);
    return 'true';
  }


}

