import { Controller, Get, Inject, Param } from '@nestjs/common';
import { ApiOkResponse, ApiTags } from '@nestjs/swagger';
import {
  PrimeLogger,
  GeoService,
  Country,
} from 'src/framework';
import { ApiDefaultResponses } from '../../decorators/api-default-responses/api-default-responses.decorator';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';

@ApiTags('geo')
@Controller('geo')
export class GeoController {
  private readonly LOGGER = new PrimeLogger(GeoController.name);
  constructor(@Inject('GeoService') private readonly geoService: GeoService) {}

  @Get('countries')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: [Country] })
  async findCountries() {
    this.LOGGER.log('Finding all countries');
    return await this.geoService.findCountries();
  }

  @Get('countries/:code')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: Country })
  async findCountryById(@Param('code') code: number) {
    this.LOGGER.log(`Finding country by code: ${code}`);
    return await this.geoService.findCountryById(code);
  }

  @Get('countries/:code/subdivisions')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: [Subdivision] })
  async findSubdivisionsByCountryCode(@Param('code') code: string) {
    this.LOGGER.log(`Finding subdivisions by country code: ${code}`);
    return await this.geoService.findSubdivisionsByCountryCode(code);
  }

  @Get('/subdivisions/:code/administrative-division')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: [Subdivision] })
  async findAdmdivisionsBySubdivisionCode(@Param('code') code: number) {
    this.LOGGER.log(
      `Finding administrative divisions by subdivision code: ${code}`,
    );
    return await this.geoService.findAdmdivisionsBySubdivisionCode(code);
  }
}
