import {
  bigint,
  boolean,
  json,
  mysqlEnum,
  mysqlTable,
  MySqlTimestampConfig,
  smallint,
  text,
  timestamp,
  varchar,
} from 'drizzle-orm/mysql-core';
import { createInsertSchema, createSelectSchema } from 'drizzle-zod';
import { HistoryMetadata, Metadata } from 'src/licitaapp/domain/entities/metadata.to';
import { Licitacion } from 'src/licitaapp/domain/type';
import { z } from 'zod';
import { RoleShortNameEnum } from '../../decorators/allow-profiles/allow-roles.decorator';
import { KeywordOrigin } from 'src/licitaapp/domain';
import { ApplicationTypeEnum } from 'src/licitaapp/domain/enum/enum.definition';
import { TypeMasterDiscriminatorEnum } from 'src/framework/domain/enum/enum.definition';

const TIMESTAMP_CONFIG: MySqlTimestampConfig = {
  mode: 'date',
};

const AUDIT_COLUMNS = {
  createdAt: timestamp('created_at', TIMESTAMP_CONFIG).notNull().defaultNow(),
  updatedAt: timestamp('updated_at', TIMESTAMP_CONFIG).onUpdateNow(),
  deletedAt: timestamp('deleted_at', TIMESTAMP_CONFIG),
  active: boolean('active').notNull().default(true),
};

export const userTable = mysqlTable('user', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  agreementAt: timestamp('agreement_at', TIMESTAMP_CONFIG),
  name: varchar('name', { length: 60 }).notNull(),
  email: varchar('email', { length: 256 }).notNull(),
  lastName: varchar('last_name', { length: 60 }),
  lastname2: varchar('last_name_2', { length: 60 }),
  cellPhone: varchar('cell_phone', { length: 20 }),
  externalId: varchar('external_id', { length: 256 }),
});

const roleShortNames = Object.values(RoleShortNameEnum) as [
  string,
  ...string[],
];


export const roleTable = mysqlTable('profile', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  name: varchar('name', { length: 60 }).notNull(),
  shortName: mysqlEnum('short_name', roleShortNames)
    .$type<RoleShortNameEnum>()
    .notNull(),
});



export const userRoleTable = mysqlTable('user_profile', {
  userId: bigint('user_id', { mode: 'number' })
    .primaryKey()
    .references(() => userTable.id),
  roleId: bigint('profile_id', { mode: 'number' })
    .primaryKey()
    .references(() => roleTable.id),
});

export const companyTable = mysqlTable('company', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  dni: varchar('dni', { length: 60 }).notNull(),
  socialReason: varchar('social_reason', { length: 256 }).notNull(),
  checkTender: boolean('check_tender').notNull().default(false),
});

export const companySubdivisionTable = mysqlTable('company_subdivision', {
  companyId: bigint('company_id', { mode: 'number' })
    .primaryKey()
    .references(() => companyTable.id),
  subdivisionId: bigint('subdivision_id', { mode: 'number' })
    .primaryKey()
    .references(() => subdivisionTable.id),
});

export const userCompanyTable = mysqlTable('user_company', {
  userId: bigint('user_id', { mode: 'number' })
    .primaryKey()
    .references(() => userTable.id),
  companyId: bigint('company_id', { mode: 'number' })
    .primaryKey()
    .references(() => companyTable.id),
});

export const typeMasterTable = mysqlTable('type_master', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  discriminator: mysqlEnum('discriminator', [
    TypeMasterDiscriminatorEnum.TUTORIAL_TYPE,
    TypeMasterDiscriminatorEnum.STATE_REQUEST_TYPE,
  ])
  .$type<TypeMasterDiscriminatorEnum>()
  .notNull(),
  shortName: varchar(('short_name'), { length: 256 }).notNull(),
  name: varchar('name', { length: 256 }).notNull(),
  order: smallint('order').default(1).notNull()
});

export const keywordTable = mysqlTable('keyword', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  value: varchar('value', { length: 256 }).notNull(),
  userId: bigint('user_id', { mode: 'number' }).notNull().references(() => userTable.id),
  origin: mysqlEnum('origin', [
    KeywordOrigin.MERCADO_PUBLICO,
    KeywordOrigin.USER,
  ])
    .$type<KeywordOrigin>()
    .default(KeywordOrigin.USER)
    .notNull(),
    companyId: bigint('company_id', { mode: 'number' })
    .notNull()
    .references(() => companyTable.id),
    metadata: json('metadata').$type<Metadata>(),
});

export const tenderTable = mysqlTable('tender', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  closeDate: timestamp('close_date', TIMESTAMP_CONFIG),
  code: varchar('code', { length: 60 }).unique().notNull(),
  name: varchar('name', { length: 256 }).notNull(),
  description: text('description').notNull(),
  subdivisionId: bigint('subdivision_id', { mode: 'number' }).references(
    () => subdivisionTable.id,
  ),
  details: json('details').$type<Licitacion>().notNull(),
  metadata: json('metadata').$type<Metadata>(),
});

export const userCompanyTenderTable = mysqlTable('user_company_tender', {
  userId: bigint('user_id', { mode: 'number' }).primaryKey(),
  companyId: bigint('company_id', { mode: 'number' }).primaryKey(),
  tenderId: bigint('tender_id', { mode: 'number' }).primaryKey(),
  ...AUDIT_COLUMNS,
  isFavorite: boolean('is_favorite').notNull().default(false),
  source: varchar('source', { length: 256 }).notNull(),
  isBuyAgile: boolean('is_buy_agile').notNull().default(false),
  matchResult: text('match_result'),
});

export const userHistoryTenderTable = mysqlTable('user_history_tender', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  userId: bigint('user_id', { mode: 'number' }).notNull().references(() => userTable.id),
  tenderId: bigint('tender_id', { mode: 'number' }).references(() => tenderTable.id),
  codeTender: varchar('code_tender', { length: 60 }),
  metadata: json('metadata').$type<HistoryMetadata>(),
});

export const continentTable = mysqlTable('continent', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  code: varchar('code', { length: 2 }).notNull(),
  name: varchar('name', { length: 16 }).notNull(),
});

export const countryTable = mysqlTable('country', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  code: varchar('code', { length: 2 }).notNull(),
  iso3: varchar('iso3', { length: 3 }).notNull(),
  number: varchar('number', { length: 3 }).notNull(),
  continentId: bigint('continent_id', { mode: 'number' }).references(
    () => continentTable.id,
  ),
  name: varchar('name', { length: 64 }).notNull(),
  fullName: varchar('full_name', { length: 128 }).notNull(),
});

export const subdivisionTable = mysqlTable('subdivision', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  code: varchar('code', { length: 12 }).notNull(),
  countryId: bigint('country_id', { mode: 'number' }).references(
    () => countryTable.id,
  ),
  name: varchar('name', { length: 256 }).notNull(),
});

export const userDeviceTable = mysqlTable('user_device', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  userId: bigint('user_id', { mode: 'number' }).notNull().references(() => userTable.id),
  token: text('token').notNull(),
  information: varchar('information', { length: 256 }).notNull(),
});

export const administrativeDivisionTable = mysqlTable(
  'administrative_division',
  {
    id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
    ...AUDIT_COLUMNS,
    code: varchar('code', { length: 12 }).notNull(),
    subdivisionId: bigint('subdivision_id', { mode: 'number' }).references(
      () => subdivisionTable.id,
    ).notNull(),
    name: varchar('name', { length: 256 }).notNull(),
  },
);

export const userRequestTable = mysqlTable('user_request', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  userId: bigint('user_id', { mode: 'number' }).notNull().references(() => userTable.id),
  companyId: bigint('company_id', { mode: 'number' }).notNull().references(() => companyTable.id),
  stateRequestTypeId: bigint('state_request_type_id', { mode: 'number' })
  .notNull().references(() => typeMasterTable.id),
  title: varchar('title', { length: 60 }).notNull(),
  description: text('description').notNull(),
  tenderId: bigint('tender_id', { mode: 'number' }).references(() => tenderTable.id),
});

export const notificationRecordTable = mysqlTable('notification_record', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  monthFilterTender: timestamp('month_filter_tender', TIMESTAMP_CONFIG),
  userId: bigint('user_id', { mode: 'number' }).notNull().references(() => userTable.id),
  tenderId: bigint('tender_id', { mode: 'number' }).references(() => tenderTable.id),
  subdivisionId: bigint('subdivision_id', { mode: 'number' }).references(() => subdivisionTable.id),
  title: varchar('title', { length: 60 }).notNull(),
  defaultMessage: varchar('default_message', { length: 256 }).notNull(),
  sendEmail: boolean('send_email').notNull().default(false),
  amountWords: bigint('amount_words', { mode: 'number' }).notNull().default(0),
  amountGeo: bigint('amount_geo', { mode: 'number' }).notNull().default(0),
  isAdmin: boolean('is_admin').notNull().default(false),
});

export const applicationLogTable = mysqlTable('application_log', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  userName: varchar('user_name', { length: 120 }).notNull(),
  statusTypeId: bigint('status_type_id', { mode: 'number' })
  .notNull().references(() => typeMasterTable.id),
  detail: text('detail').notNull(),
  type: mysqlEnum('type', [
    ApplicationTypeEnum.CHECK_TENDER_FAVORITES_TO_CLOSE_LOG,
    ApplicationTypeEnum.DAILY_EMAIL_SUMMARY_LOG,
    ApplicationTypeEnum.CHECK_NEW_TENDERS_USER_LOG,
    ApplicationTypeEnum.METADATA_TASK_LOG,
    ApplicationTypeEnum.KEYWORDS_LOG,
    ApplicationTypeEnum.TENDER_WITHOUT_META_LOG,
    ApplicationTypeEnum.TENDER_COMPLETE_STATE_LOG,
    ApplicationTypeEnum.TENDER_CHECK_CLOSE_DATE_LOG,
    ApplicationTypeEnum.TENDER_DELETE_CLOSE_DATE_LOG,
    ApplicationTypeEnum.TENDER_CHECK_SUBDIVISION_LOG,
    ApplicationTypeEnum.SUMMARY_ADMIN_LOG,
    ApplicationTypeEnum.RECALCULATE_TENDER_DASHBOARD_LOG,
    ApplicationTypeEnum.SEARCH_TENDERS_MERCADO_PUBLICO_LOG,
    ApplicationTypeEnum.ERRASE_OLD_TENDERS_LOG,
  ])
  .$type<ApplicationTypeEnum>()
  .notNull()
});

export const userCampaignTable = mysqlTable('user_campaign', {
  id: bigint('id', { mode: 'number' }).autoincrement().primaryKey(),
  ...AUDIT_COLUMNS,
  description: varchar('description', { length: 60 }).notNull(),
  title: varchar('title', { length: 60 }).notNull(),
  defaultMessage: varchar('default_message', { length: 256 }).notNull(),
  dayOfWeek: varchar('day_of_week', { length: 60 }).notNull(), // e.g., '1-2-3-4' for Monday to Thursday
  defHour: smallint('def_hour'),
  defMinute: smallint('def_minute'),
});

export const applicationLogTableInsertSchema = createInsertSchema(
  applicationLogTable,
);
export const applicationLogTableUpdateSchema = applicationLogTableInsertSchema.partial();
export const applicationLogTableSelectSchema = createSelectSchema(
  applicationLogTable,
).pick({
  id: true,
  userName: true,
  statusTypeId: true,
  detail: true,
  type: true,
});

export const notificationRecordTableInsertSchema = createInsertSchema(
  notificationRecordTable,
);
export const notificationRecordTableUpdateSchema = notificationRecordTableInsertSchema.partial();
export const notificationRecordTableSelectSchema = createSelectSchema(
  notificationRecordTable,
).pick({
  id: true,
  userId: true,
  title: true,
  defaultMessage: true,
  tenderId: true,
  subdivisionId: true,
  sendEmail: true,
  amountWords: true,
  amountGeo: true,
  isAdmin: true,
});

export const userTableInsertSchema = createInsertSchema(userTable);
export const userTableUpdateSchema = userTableInsertSchema.partial();
export const userTableSelectSchema = createSelectSchema(userTable).pick({
  id: true,
  name: true,
  email: true,
  lastName: true,
  lastname2: true,
  cellPhone: true,
});

export const roleTableInsertSchema = createInsertSchema(roleTable);
export const roleTableUpdateSchema = roleTableInsertSchema.partial();
export const roleTableSelectSchema = createSelectSchema(roleTable).pick({
  id: true,
  name: true,
  shortName: true,
});

export const typeMasterTableInsertSchema = createInsertSchema(typeMasterTable);
export const typeMasterTableUpdateSchema = typeMasterTableInsertSchema.partial();
export const typeMasterTableSelectSchema = createSelectSchema(typeMasterTable).pick({
  id: true,
  name: true,
  shortName: true,
  discriminator: true,
  order: true,
});

export const userRoleTableInsertSchema =
  createInsertSchema(userRoleTable);
export const userRoleTableUpdateSchema =
  userRoleTableInsertSchema.partial();
export const userRoleTableSelectSchema = createSelectSchema(
  userRoleTable,
).pick({
  userId: true,
  roleId: true,
});

export const companyTableInsertSchema = createInsertSchema(companyTable);
export const companyTableUpdateSchema = companyTableInsertSchema.partial();
export const companyTableSelectSchema = createSelectSchema(companyTable).pick({
  id: true,
  dni: true,
  socialReason: true,
  checkTender: true,
});

export const userCompanyTableInsertSchema =
  createInsertSchema(userCompanyTable);
export const userCompanyTableUpdateSchema =
  userCompanyTableInsertSchema.partial();
export const userCompanyTableSelectSchema = createSelectSchema(
  userCompanyTable,
).pick({
  userId: true,
  companyId: true,
});

export const keywordTableInsertSchema = createInsertSchema(keywordTable);
export const keywordTableUpdateSchema = keywordTableInsertSchema.partial();
export const keywordTableSelectSchema = createSelectSchema(keywordTable).pick({
  id: true,
  value: true,
  origin: true,
  companyId: true,
});

export const userCompanyTenderTableInsertSchema = createInsertSchema(userCompanyTenderTable);
export const userCompanyTenderTableUpdateSchema = userCompanyTenderTableInsertSchema.partial();
export const userCompanyTenderTableSelectSchema = createSelectSchema(userCompanyTenderTable).pick({
  userId: true,
  companyId: true,
  tenderId: true,
  isBuyAgile: true,
  source: true,
  matchResult: true,
  isFavorite: true,
});

export const userHistoryTenderTableInsertSchema = createInsertSchema(userHistoryTenderTable);
export const userHistoryTenderTableUpdateSchema = userHistoryTenderTableInsertSchema.partial();
export const userHistoryTenderTableSelectSchema = createSelectSchema(userHistoryTenderTable).pick({
  userId: true,
  tenderId: true,  
});

export const tenderTableInsertSchema = createInsertSchema(tenderTable, {
  details: z.any(),
});
export const tenderTableUpdateSchema = tenderTableInsertSchema.partial();
export const tenderTableSelectSchema = createSelectSchema(tenderTable).pick({
  id: true,
  code: true,
  name: true,
  description: true,
  subdivisionId: true,
  details: true,
});

export const userDeviceTableInsertSchema = createInsertSchema(userDeviceTable);
export const userDeviceTableUpdateSchema = userDeviceTableInsertSchema.partial();
export const userDeviceTableSelectSchema = createSelectSchema(userDeviceTable).pick({
  id: true,
  userId: true,
  token: true,
});

export const userRequestTableInsertSchema = createInsertSchema(userRequestTable);
export const userRequestTableUpdateSchema = userRequestTableInsertSchema.partial();
export const userRequestTableSelectSchema = createSelectSchema(userRequestTable).pick({
  id: true,
  userId: true,
  companyId: true,
  tenderId: true,
  stateRequestTypeId: true,
});

export const userCampaignTableInsertSchema = createInsertSchema(userCampaignTable);
export const userCampaignTableUpdateSchema = userCampaignTableInsertSchema.partial();
export const userCampaignTableSelectSchema = createSelectSchema(userCampaignTable).pick({
  id: true,
  title: true,
  defaultMessage: true,
  dayOfWeek: true,
  defHour: true,
  defMinute: true,
  description: true,
});
