import { Injectable } from '@nestjs/common';
import { NotificationRecordRepository } from 'src/framework/application/repository/notification-record-repository/notification-record-repository.interface';
import { NotificationRecord, InsertNotificationRecord, UpdateNotificationRecord } from 'src/framework/domain';
import { DBConfigService, TxType } from '../../drizzle';
import { PrimeLogger } from '../../definition';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { and, desc, eq, inArray } from 'drizzle-orm';

@Injectable()
export class NotificationRecordRepositoryImpl implements NotificationRecordRepository {
    private readonly LOGGER = new PrimeLogger(NotificationRecordRepositoryImpl.name);
    constructor(private readonly db: DBConfigService) {}
    async erraseNotificationRecordsByTender(tenderIds: number[], tx?: TxType): Promise<void> {
        this.LOGGER.log(`Erasing notification records for tenders with IDs: ${tenderIds.join(', ')}`);
        await (tx || this.db.conn)
          .delete(schema.notificationRecordTable)
          .where(inArray(schema.notificationRecordTable.tenderId, tenderIds))
          .execute();
    }
    async markNotificationEmailAsSent(notificationRecordId: number[], tx?: TxType){
        this.LOGGER.log(`markNotificationEmailAsSent notificationRecordId: ${notificationRecordId}`);
        await (tx || this.db.conn)
          .update(schema.notificationRecordTable)
          .set({ sendEmail: true, updatedAt: TenderUtil.getCurrentSystemDate() })
          .where(inArray(schema.notificationRecordTable.id, notificationRecordId));
    }

    async saveAll(notificationRecords: InsertNotificationRecord[], tx?: TxType) {
        this.LOGGER.log(`saveAll notificationRecords size: ${notificationRecords.length}`);
        const validatedNotificationRecord = notificationRecords.map((notificationRecord) =>
          schema.notificationRecordTableInsertSchema.parse(notificationRecord),
        );
        await (tx || this.db.conn)
          .insert(schema.notificationRecordTable)
          .values(validatedNotificationRecord)
          .execute();
      }

    async updateNotification(notificationRecordId: number, notificationRecord: UpdateNotificationRecord, tx?: TxType) {
        this.LOGGER.log(`updateNotification notificationRecord: ${notificationRecordId}`);    
        await (this.db.conn)
          .update(schema.notificationRecordTable)
          .set({ 
            updatedAt: TenderUtil.getCurrentSystemDate(), 
            active: notificationRecord.active, 
          })
          .where(eq(schema.notificationRecordTable.id, notificationRecordId));
    }

    async getActiveNotificationRecords(userId:number, isAdmin:boolean = false): Promise<NotificationRecord[]> {
        this.LOGGER.log(`getActiveNotificationRecords userId: ${userId}`);
        
        if(isAdmin){
            return await this.db.conn
            .select({
                id: schema.notificationRecordTable.id,
                userId: schema.notificationRecordTable.userId,
                title: schema.notificationRecordTable.title,
                defaultMessage: schema.notificationRecordTable.defaultMessage,
                active: schema.notificationRecordTable.active,
                createdAt: schema.notificationRecordTable.createdAt,
                updatedAt: schema.notificationRecordTable.updatedAt,
                monthFilterTender: schema.notificationRecordTable.monthFilterTender,
                tenderId: schema.notificationRecordTable.tenderId,
            })
            .from(schema.notificationRecordTable)
            .where(and(
                eq(schema.notificationRecordTable.active, true),
                eq(schema.notificationRecordTable.isAdmin, isAdmin))
            ).then((rows) => {
                return rows.map((row) => new NotificationRecord(row.id, row.userId, row.title, row.defaultMessage, 
                    row.active, row.createdAt, row.updatedAt, row.tenderId?row.tenderId:undefined, undefined, row.monthFilterTender));
            });
        }        
        return await this.db.conn
        .select({
            id: schema.notificationRecordTable.id,
            userId: schema.notificationRecordTable.userId,
            title: schema.notificationRecordTable.title,
            defaultMessage: schema.notificationRecordTable.defaultMessage,
            active: schema.notificationRecordTable.active,
            createdAt: schema.notificationRecordTable.createdAt,
            updatedAt: schema.notificationRecordTable.updatedAt,
            monthFilterTender: schema.notificationRecordTable.monthFilterTender,
            tenderId: schema.notificationRecordTable.tenderId,
        })
        .from(schema.notificationRecordTable)
        .where(and(
            eq(schema.notificationRecordTable.userId, userId), 
            eq(schema.notificationRecordTable.active, true),
            eq(schema.notificationRecordTable.isAdmin, isAdmin))
        ).then((rows) => {
            return rows.map((row) => new NotificationRecord(row.id, row.userId, row.title, row.defaultMessage, 
                row.active, row.createdAt, row.updatedAt, row.tenderId?row.tenderId:undefined, undefined, row.monthFilterTender));
        });
    }

    async getNotificationRecordsPaginated(userId:number, page:number, pageSize:number){
        this.LOGGER.log(`getNotificationRecordsPaginated userId: ${userId} page: ${page} pageSize: ${pageSize}`);
        return await this.db.conn
        .select({
            id: schema.notificationRecordTable.id,
            userId: schema.notificationRecordTable.userId,
            title: schema.notificationRecordTable.title,
            defaultMessage: schema.notificationRecordTable.defaultMessage,
            active: schema.notificationRecordTable.active,
            createdAt: schema.notificationRecordTable.createdAt,
            updatedAt: schema.notificationRecordTable.updatedAt,
        })
        .from(schema.notificationRecordTable)
        .where(eq(schema.notificationRecordTable.userId, userId))
        .orderBy(desc(schema.notificationRecordTable.createdAt))
        .limit(Number(pageSize))
        .offset(pageSize * (page - 1))
        .then((rows) => {
            return rows.map((row) => new NotificationRecord(row.id, row.userId, row.title, row.defaultMessage, row.active, row.createdAt, row.updatedAt));
        });

    }

    async logicalRemoveBYListId(ids: number[], tx?: TxType) {
        this.LOGGER.log(`logicalRemoveBYListId ids: ${ids}`);
        await (tx || this.db.conn)
            .update(schema.notificationRecordTable)
            .set({ active: false, deletedAt: TenderUtil.getCurrentSystemDate() })
            .where(inArray(schema.notificationRecordTable.id, ids));
    }

    async getAdminNotification(): Promise<NotificationRecord[]> {
        this.LOGGER.log(`getAdminNotification`);
        return await this.db.conn
        .select({
            id: schema.notificationRecordTable.id,
            userId: schema.notificationRecordTable.userId,
            title: schema.notificationRecordTable.title,
            defaultMessage: schema.notificationRecordTable.defaultMessage,
        })
        .from(schema.notificationRecordTable)
        .where(and(eq(schema.notificationRecordTable.sendEmail, false), eq(schema.notificationRecordTable.isAdmin, true)))
        .then((rows) => {
            return rows.map((row) => {
                return new NotificationRecord(row.id, row.userId, row.title, row.defaultMessage, undefined, undefined, undefined, undefined, undefined, undefined
                    , undefined
                );
            });
        });
    }

    async getUserNotificationRecordsActive(): Promise<NotificationRecord[]> {
        this.LOGGER.log(`getUserNotificationRecordsActive`);
        return await this.db.conn
        .select({
            id: schema.notificationRecordTable.id,
            userId: schema.notificationRecordTable.userId,
            title: schema.notificationRecordTable.title,
            defaultMessage: schema.notificationRecordTable.defaultMessage,
            amountWords: schema.notificationRecordTable.amountWords,
            amountGeo: schema.notificationRecordTable.amountGeo,
            userEmail: schema.userTable.email,
        })
        .from(schema.notificationRecordTable)
        .leftJoin(schema.userTable, eq(schema.notificationRecordTable.userId, schema.userTable.id))
        .where(and(eq(schema.notificationRecordTable.sendEmail, false)))
        .then((rows) => {
            return rows.map((row) => {
                const element = new NotificationRecord(row.id, row.userId, row.title, row.defaultMessage, undefined, undefined, undefined, undefined, undefined, undefined
                    , undefined, row.amountWords, row.amountGeo
                );
                element.userEmail = row.userEmail;
                return element;
            });
        });
    }
}
