import { Injectable } from '@nestjs/common';
import { UserRequestRepository } from 'src/framework/application/repository/user-request-repository/user-request-repository.interface';
import {
  InsertUserRequest,
  RequestTenderTO,
  UpdateUserRequest,
  UserRequest,
} from 'src/framework/domain';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';
import { DBConfigService } from '../../drizzle';
import { TxType } from '../../drizzle/drizzle.types';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';
import { eq, and, desc, inArray } from 'drizzle-orm';
import { inspect } from 'util';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { TenderTO } from 'src/licitaapp/domain';

@Injectable()
export class UserRequestRepositoryImpl implements UserRequestRepository {
  private readonly LOGGER = new PrimeLogger(UserRequestRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async erraseUserRequests(tenderIds: TenderTO[]): Promise<void> {
    this.LOGGER.log(`Erasing user requests for tender IDs: ${tenderIds.join(', ')}`);
    await this.db.conn
        .delete(schema.userRequestTable)
        .where(inArray(schema.userRequestTable.tenderId, tenderIds.map(t => t.tenderId)))
        .execute();
    
  }
  async save(
    dataInsert: InsertUserRequest,
    userId: number,
    tx?: TxType,
  ): Promise<UserRequest> {
    this.LOGGER.log(`save repository: ${inspect(dataInsert)}`);
    const validatedUserRequest = schema.userRequestTableInsertSchema.parse({
      userId: userId,
      tenderId: dataInsert.tenderId,
      companyId: dataInsert.companyId,
      stateRequestTypeId: 5,
      title: dataInsert.title,
      description: dataInsert.description,
    });
    const insertedUserRequestId = await (tx || this.db.conn)
      .insert(schema.userRequestTable)
      .values(validatedUserRequest)
      .$returningId()
      .then((rows) => {
        return rows[0].id;
      });

    return await this.findById(insertedUserRequestId, tx).then((dataObject) => {
      if (dataObject) {
        return dataObject;
      } else {
        throw new Error('UserRequest not found');
      }
    });
  }

  async existUserRequest(
    userId: number,
    tenderId: number,
    companyId: number,
  ): Promise<number> {
    this.LOGGER.log(`existUserRequest repository: ${userId}`);
    return await this.db.conn
      .select({
        id: schema.userRequestTable.id,
      })
      .from(schema.userRequestTable)
      .where(
        and(
          eq(schema.userRequestTable.userId, userId),
          eq(schema.userRequestTable.tenderId, tenderId),
          eq(schema.userRequestTable.companyId, companyId),
        ),
      )
      .then((rows) => {
        if (rows.length === 0) {
          return 0;
        }
        return rows[0].id;
      });
  }

  async findById(id: number, tx?: TxType): Promise<UserRequest | null> {
    this.LOGGER.log(`findById repository: ${id}`);
    return await (tx || this.db.conn)
      .select({
        id: schema.userRequestTable.id,
        userId: schema.userRequestTable.userId,
        tenderId: schema.userRequestTable.tenderId,
        companyId: schema.userRequestTable.companyId,
        stateRequestTypeId: schema.userRequestTable.stateRequestTypeId,
        createdAt: schema.userRequestTable.createdAt,
        updatedAt: schema.userRequestTable.updatedAt,
        title: schema.userRequestTable.title,
        description: schema.userRequestTable.description,
      })
      .from(schema.userRequestTable)
      .where(eq(schema.userRequestTable.id, id))
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }
        let newValue = new UserRequest(
          rows[0].id,
          rows[0].companyId,
          rows[0].stateRequestTypeId,
          rows[0].title,
          rows[0].tenderId,
          rows[0].description,
          rows[0].userId,
        );
        newValue.createdAt = rows[0].createdAt;
        newValue.updatedAt = rows[0].updatedAt;
        return newValue;
      });
  }

  async update(
    userRequestId: number,
    objectUpdateData: UpdateUserRequest,
    tx?: TxType,
  ): Promise<UserRequest | null> {
    this.LOGGER.log(`update repository: ${userRequestId}`);
    const validatedUserRequest = schema.userRequestTableUpdateSchema.parse({
      stateRequestTypeId: objectUpdateData.stateRequestTypeId,
      updatedAt: TenderUtil.getCurrentSystemDate(),
    });
    await (tx || this.db.conn)
      .update(schema.userRequestTable)
      .set(validatedUserRequest)
      .where(eq(schema.userRequestTable.id, userRequestId))
      .execute();
    return await this.findById(userRequestId, tx).then((dataObject) => {
      if (dataObject) {
        return dataObject;
      } else {
        throw new Error('UserRequest not found');
      }
    });
  }

  async getPagination(
    page: number,
    pageSize: number,
    shortStateUserRequest: string,
    filter: string,
  ): Promise<RequestTenderTO[]> {
    const offset = (page - 1) * pageSize;
    const selectFields = {
      field1: schema.userRequestTable.id,
      field2: schema.companyTable.socialReason,
      field3: schema.companyTable.dni,
      field4: schema.tenderTable.code,
      field5: schema.userTable.name,
      field6: schema.userTable.lastName,
      field7: schema.userRequestTable.createdAt,
      field8: schema.userRequestTable.updatedAt,
      field9: schema.typeMasterTable.name,
      field10: schema.userTable.email,
      field11: schema.userTable.cellPhone,
    };
    const query = this.db.conn
      .select(selectFields)
      .from(schema.userRequestTable)
      .leftJoin(
        schema.typeMasterTable,
        eq(
          schema.typeMasterTable.id,
          schema.userRequestTable.stateRequestTypeId,
        ),
      )
      .leftJoin(
        schema.tenderTable,
        eq(schema.tenderTable.id, schema.userRequestTable.tenderId),
      )
      .leftJoin(
        schema.companyTable,
        eq(schema.companyTable.id, schema.userRequestTable.companyId),
      )
      .leftJoin(
        schema.userTable,
        eq(schema.userTable.id, schema.userRequestTable.userId),
      )
      .where(
        and(
          eq(schema.typeMasterTable.shortName, shortStateUserRequest),
          eq(schema.userRequestTable.active, true),
        ),
      )
      .orderBy(desc(schema.userRequestTable.updatedAt))
      .limit(Number(pageSize))
      .offset(offset);
    const rows = await query;

    if (rows.length === 0) {
      return [];
    }

    return rows.map(
      (row) =>
        new RequestTenderTO(
          row.field1,
          row.field2 ? row.field2 : '',
          row.field3 ? row.field3 : '',
          row.field4 ? row.field4 : '',
          this.evalFullName(row.field5, row.field6),
          row.field9 ? row.field9 : '',
          row.field11 ? row.field11 : '',
          row.field10 ? row.field10 : '',
          row.field7 ? row.field7 : null,
          row.field8 ? row.field8 : null,
        ),
    );
  }

  private evalFullName(name: string | null, lastName: string | null): string {
    if (lastName) {
      return `${name} ${lastName}`;
    }
    return name ? name : '';
  }

  async logicalRemove(userRequestId: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`logicalRemove repository: ${userRequestId}`);
    await (tx || this.db.conn)
      .update(schema.userRequestTable)
      .set({ active: false, deletedAt: TenderUtil.getCurrentSystemDate() })
      .where(and(eq(schema.userRequestTable.id, userRequestId)))
      .execute();
  }

  async updateState(
    userRequestId: number,
    stateRequestTypeId: number,
    active: boolean,
    tx?: TxType,
  ): Promise<UserRequest | null> {
    this.LOGGER.log(`update repository: ${userRequestId}`);
    const validatedUserRequest = schema.userRequestTableUpdateSchema.parse({
      stateRequestTypeId: stateRequestTypeId,
      updatedAt: TenderUtil.getCurrentSystemDate(),
    });
    if (!active) {
      (validatedUserRequest.active = active),
        (validatedUserRequest.deletedAt = TenderUtil.getCurrentSystemDate());
    }
    await (tx || this.db.conn)
      .update(schema.userRequestTable)
      .set(validatedUserRequest)
      .where(eq(schema.userRequestTable.id, userRequestId))
      .execute();
    return await this.findById(userRequestId, tx).then((dataObject) => {
      if (dataObject) {
        return dataObject;
      } else {
        throw new Error('UserRequest not found');
      }
    });
  }
}
