import { Inject, Injectable } from '@nestjs/common';
import { EmailService, FirebaseService, UserDeviceService, UserService } from 'src/framework/application';
import * as admin from 'firebase-admin';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';
import { NotificationRecordService } from 'src/framework/application/service/notification-record-service/notification-record-service.interface';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { NotificationRecord } from 'src/framework/domain';
type NotificationData = { [key: string]: string };

@Injectable()
export class FirebaseImplService implements FirebaseService {

  private readonly LOGGER = new PrimeLogger(FirebaseImplService.name);
  private readonly messaging: admin.messaging.Messaging;
  private readonly auth: admin.auth.Auth;
  
  constructor(
    @Inject('NotificationRecordService')
    private readonly notificationRecordService: NotificationRecordService,
    @Inject('UserDeviceService')
    private readonly userDeviceService: UserDeviceService,
    
    @Inject('EmailService')
    private readonly emailService: EmailService,
  ) {
    if (admin.apps.length === 0) {
      admin.initializeApp({
        credential: admin.credential.applicationDefault(),
      });
    }
    this.messaging = admin.messaging();
    this.auth = admin.auth();
  }
  async verifyIdToken(idToken: string): Promise<admin.auth.DecodedIdToken> {
    try {
      const decodedToken = await this.auth.verifyIdToken(idToken);
      this.LOGGER.log(`Token verificado: ${JSON.stringify(decodedToken)}`);
      return decodedToken;
    } catch (error) {
      this.LOGGER.error(`Error verificando el token: ${error.message}`);
      throw error;
    }
  }

  stringToNumberHash(str: string): number {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      hash = (hash * 31 + str.charCodeAt(i)) % 1000000007;
    }
    return hash;
  }

  async sendPushNotification(
    tokens: string[],
    title: string,
    body: string,
    data?: NotificationData,
  ) {
    if (tokens.length === 0) {
      this.LOGGER.warn('No tokens to send message');
      return;
    }
    const currentDate = new Date();
    const idMessage = this.stringToNumberHash(title + currentDate.getDay());
    data = { ...data, id_message: idMessage.toString() };
    const messages: admin.messaging.Message[] = tokens.map((token) => ({
      notification: {
        title,
        body,
      },
      data,
      token,
    }));

    try {
      const response = await this.messaging.sendEach(messages);
      for (const res of response.responses) {
        if (res.success) {
          this.LOGGER.debug(
            `Successfully sent message: ${JSON.stringify(response)}`
          );
        } else {
          const index = response.responses.indexOf(res);
          const token = tokens[index];
          const jsonTxt = JSON.stringify(res);
          this.LOGGER.error(
            `Error sending message to token ${token} (index ${index}): ${jsonTxt}`
          );
          if(jsonTxt.includes('messaging/registration-token-not-registered')) {
            this.userDeviceService.erraseUserDevice(token);
          }
          
        }
      }
      return response;
    } catch (error) {
      this.LOGGER.error(
        `Error sending message: ${JSON.stringify(error)}`
      );
      throw error;
    }
  }

  async deleteUserFromFirebase(uid: string): Promise<void> {
    this.LOGGER.log(`Eliminando usuario con UID ${uid} de Firebase`);
    try {
      await this.auth.deleteUser(uid);
      this.LOGGER.log(`Usuario con UID ${uid} eliminado de Firebase`);
    } catch (error) {
      this.LOGGER.error(`Error eliminando el usuario con UID ${uid}`, error);
      throw error;
    }
  }

  async groupNotifications(userId: number) {
    this.LOGGER.log(`groupNotifications - userId: ${userId}`);
    const tokens = await this.userDeviceService.findTokensByUserId(userId);

    if (tokens.length === 0) {
      this.LOGGER.log(`groupNotifications - no tokens for user: ${userId}`);
      return;
    }

    const listRecordsToNotify = await this.notificationRecordService.getActiveNotificationRecords(userId);

    const counts = listRecordsToNotify.reduce((acc, record) => {
      if (record.title.includes('Encontradas por Ubicación')) acc.countForSubdivision++;
      else if (record.title.includes('favorita por cerrar')) acc.countForFavoriteToClose++;
      else if (record.monthFilterTender) acc.countForMonth++;
      return acc;
    }, { countForSubdivision: 0, countForFavoriteToClose: 0, countForMonth: 0 });

    this.LOGGER.log(`groupNotifications - tokens: ${tokens.length} for user: ${userId} countForFavoriteToClose: ${counts.countForFavoriteToClose} countForSubdivision: ${counts.countForSubdivision} countForMonth: ${counts.countForMonth}`);

    const notifications = [
      { condition: counts.countForSubdivision > 0, title: 'Encontradas por Ubicación', body: `Licitaciones encontradas para tus regiones de interés`, type_message: 'general_tender_alert' },
      { condition: counts.countForFavoriteToClose > 0, title: 'Favoritos por cerrarse', body: `Licitaciones favoritas por cerrarse encontradas ${counts.countForFavoriteToClose}`, type_message: 'favorite_tender_alert' },
      { condition: counts.countForMonth > 0, title: 'Encontradas por Rubros', body: 'Licitaciones encontradas por palabras claves', type_message: 'history_tender' }
    ];

    try{
      for (const notification of notifications) {
        if (notification.condition) {
          await this.sendPushNotification(tokens, notification.title, notification.body, { type_message: notification.type_message });
        }
      }
    }catch(error){
      this.LOGGER.error(`groupNotifications try to send to users - Error: ${error}`);
      throw error;
    }

    this.notificationRecordService.logicalRemoveBYListId(listRecordsToNotify.map(record => record.id));
    this.notifyToAdminUsers();
  }

  async notifyToAdminUsers(){
    const tokensUser = await this.userDeviceService.getAdminUserDeviceTokens();
    
    if(tokensUser.length === 0){
      this.LOGGER.log(`notifyToAdminUsers - no tokens for admin`);
      return;
    }

    const listRecordsToNotify = await this.notificationRecordService.getActiveNotificationRecords(0, true);
    this.LOGGER.log(`notifyToAdminUsers - listRecordsToNotify: ${listRecordsToNotify.length}`);
    if (listRecordsToNotify.length>0) {
      try{
        if(listRecordsToNotify.length==1){
          //FIX: arreglar aca yuri
          //const user = await this.userService.getUserById(listRecordsToNotify[0].userId);
          //const fullName = user && `${user.name} ${user.lastName != null ? user.lastName: ''} ${user.lastname2!=null ? user.lastname2: ''}`.trim();
          await this.sendPushNotification(tokensUser, 'Administrador', `El usuario  ha solicitado apoyo por una licitación.`, { type_message: 'tender_new' });
        }else{
          await this.sendPushNotification(tokensUser, 'Administrador', `Existen ${listRecordsToNotify.length} solicitudes por responder.`, { type_message: 'tender_new' });
        }
  
        this.notificationRecordService.logicalRemoveBYListId(listRecordsToNotify.map(record => record.id));
      }catch(error){
        this.LOGGER.error(`notifyToAdminUsers - Error: ${error}`);
        throw error;
      }
      
    }

  }

  async sendDailyEmailSumary(applicationLogId: number) {
    this.LOGGER.log(`sendDailyEmailSumary - START - applicationLogId: ${applicationLogId}`);
    const sendToNotify = await this.notificationRecordService.getUserNotificationRecordsActive();

    const messageTxt = `sendDailyEmailSumary - applicationLogId: ${applicationLogId} - sendToNotify size: ${sendToNotify.length}`;
    const groupedNotifications = sendToNotify.reduce((acc: { [key: number]: NotificationRecord[] }, notification) => {
      if (!acc[notification.userId]) {
        acc[notification.userId] = [];
      }
      acc[notification.userId].push(notification);
      return acc;
    }, {});

    const systemHour = TenderUtil.getSystemDateDDMMYYYY();
    const dayOfWeek = systemHour.toLocaleString('es-CL', { weekday: 'long' });

    for (const userId in groupedNotifications) {
      messageTxt.concat(`. Processing userId: ${userId}`);
      const notifications = groupedNotifications[userId];
      const totalAmountWords = notifications.reduce((sum, notification) => sum + (notification.amountWords || 0), 0);
      const totalGeo = notifications.reduce((sum, notification) => sum + (notification.amountGeo || 0), 0);
      this.LOGGER.log(`Processing notifications email line for user: ${userId} - totalAmountWords: ${totalAmountWords} - totalGeo: ${totalGeo}`);
      
      await this.sendNewsEmail(notifications[0].userEmail!, `${dayOfWeek} ${systemHour.toLocaleDateString('es-CL', { day: '2-digit', month: '2-digit', year: 'numeric' })} `, totalAmountWords, totalGeo);
      
      this.notificationRecordService.markNotificationEmailAsSent(notifications.map(notification => notification.id));
    }
    return messageTxt;
  }

  async sendNewsEmail(email:string, dayOfweekLabel: string, amountWordsTender: number=0, amountGeoTenders: number = 0) {
    this.LOGGER.log(`Sending news email to: ${email}`);
    await this.emailService.sendNewsEmail(
      email,
      'LICITAAPP Resumen diario actividades',
      dayOfweekLabel,
      amountWordsTender,
      amountGeoTenders
    );
  }

  async sendEmailSumaryToAdmin(applicationLogId:number, userService: UserService){
    const userIdsAdmin = await userService.findEmailsAdmin();
    var messageTxt = `ApplicationLogId: ${applicationLogId} - Administradores activos: ${userIdsAdmin.length}`;
    this.LOGGER.log(`sendEmailSumaryToAdmin - userIdsAdmin: ${userIdsAdmin.length}`);
    const sendToNotify = await this.notificationRecordService.getAdminNotification();
    this.LOGGER.log(`sendEmailSumaryToAdmin - sendToNotify: ${sendToNotify.length}`);
    if(sendToNotify.length > 0){
      for(const email of userIdsAdmin){
        messageTxt = messageTxt.concat(` - Enviando email a: ${email}`);
        await this.emailService.sendAdminEmail(email, 'LICITAAPP Administrador', `Administrador, existen ${sendToNotify.length} solicitudes por responder.`);
      }
      messageTxt = messageTxt.concat(` - Enviando notificaciones: ${sendToNotify.length}`);
      this.notificationRecordService.markNotificationEmailAsSent(sendToNotify.map(notification => notification.id));
      messageTxt = messageTxt.concat(` - Notificacion finalizada.`);
    }
    return messageTxt;
  }
  
}