import { HttpService } from '@nestjs/axios';
import { Injectable } from '@nestjs/common';
import { firstValueFrom, map } from 'rxjs';
import { convertDates } from './mp.util';
import {
  LicitacionesQueryParams,
  LicitacionesQueryParamsSchema,
  LicitacionesResponse,
  ListaEmpresasResponse,
  OrdenesDeCompraQueryParams,
  OrdenesDeCompraQueryParamsSchema,
  OrdenesDeCompraResponse,
} from './types';
import { PrimeLogger } from 'src/framework';

@Injectable()
export class MercadoPublicoRepository {
  private readonly LOGGER = new PrimeLogger(MercadoPublicoRepository.name);
  constructor(
    private readonly httpService: HttpService
  ) {}

  async licitaciones(
    args: LicitacionesQueryParams,
  ): Promise<LicitacionesResponse | undefined> {
    LicitacionesQueryParamsSchema.parse(args);

    const response = await firstValueFrom(
      this.httpService
        .get<LicitacionesResponse>('/licitaciones.json', {
          params: args,
        })
        .pipe(
          map((response) =>
            LicitacionesResponse.fromJson(convertDates(response.data)),
          ),
        ),
    ).catch((error) => {
      this.LOGGER.error(
        `Error retrieving licitaciones. ${error.message}. Params: ${JSON.stringify(args)}`,
        'MercadoPublicoRepository',
      );
      throw error;
    });
    return response;
  }

  async ordenesDeCompra(
    args: OrdenesDeCompraQueryParams,
  ): Promise<OrdenesDeCompraResponse | undefined> {
    OrdenesDeCompraQueryParamsSchema.parse(args);

    const response = await firstValueFrom(
      this.httpService
        .get<OrdenesDeCompraResponse>('/ordenesdecompra.json', {
          params: args,
        })
        .pipe(
          map((response) =>
            OrdenesDeCompraResponse.fromJson(convertDates(response.data)),
          ),
        ),
    ).catch((error) => {
      this.LOGGER.error(
        `Error retrieving ordenes de compra. ${error.message}. Params: ${JSON.stringify(args)}`,
        'MercadoPublicoRepository',
      );
      throw error;
    });
    return response;
  }

  async buscarProveedor(
    rut: string,
  ): Promise<ListaEmpresasResponse | undefined> {
    const response = await firstValueFrom(
      this.httpService
        .get<ListaEmpresasResponse>('/empresas/BuscarProveedor', {
          params: {
            rutempresaproveedor: rut,
          },
        })
        .pipe(map((response) => response.data)),
    ).catch((error) => {
      this.LOGGER.error(
        `Error: ${JSON.stringify(error)}`,
        'MercadoPublicoRepository',
      );
      return undefined;
    });
    return response;
  }

  async buscarComprador(): Promise<ListaEmpresasResponse | undefined> {
    const response = await firstValueFrom(
      this.httpService
        .get<ListaEmpresasResponse>('/empresas/BuscarComprador')
        .pipe(map((response) => response.data)),
    ).catch((error) => {
      this.LOGGER.error(
        `Error: ${JSON.stringify(error)}`,
        'MercadoPublicoRepository',
      );
      return undefined;
    });
    return response;
  }
}
