import {
  Body,
  Controller,
  Delete,
  Get,
  Inject,
  Param,
  Post,
  Put,
} from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import {
  AllowRoles,
  RoleShortNameEnum,
} from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import {
  CompanyService,
} from 'src/licitaapp';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { ActionReview, CompanyFullTO, CompanyRegisterTO, DashboardTO, HistoryTender, HistoryTenderDetail, Tender } from 'src/licitaapp/domain';

@ApiTags('tenders')
@Controller('tenders')
export class TendersController {
  private readonly LOGGER = new PrimeLogger(TendersController.name);
  constructor(
    @Inject('TenderService') private tenderService: TenderService,
    @Inject('CompanyService') private companyService: CompanyService,
  ) {}
  @Put('review-tender-info')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  @ApiBody({ type: ActionReview })
  async reviewTendersInfo(@Body() paramTO: ActionReview): Promise<String> {
    this.LOGGER.log(`init reviewTendersInfo action: ${paramTO}`);
    this.tenderService.reviewTendersInfo(paramTO.action);
    return 'true';
  }

  @Post('create-company')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: CompanyFullTO })
  @ApiBody({ type: CompanyRegisterTO })
  async createCompany(
    @PrimeUser() user: User,
    @Body() company: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.warn(`Creating company for user: ${user.id}`);
    const newCompany = await this.companyService.createCompany(
      user.id,
      company,
    );
    newCompany?.company.id &&
      this.tenderService.createDashboardCompany(
        newCompany?.company.id,
        user.id,
      );
    return newCompany;
  }

  @Put('create-dashboard-company/:companyId')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  @ApiParam({ name: 'companyId', type: 'number' })
  async createDashboardCompany(
    @PrimeUser() user: User,
    @Param('companyId') companyId: number,
  ): Promise<String> {
    this.LOGGER.log(
      `init createDashboardCompany companyId: ${companyId} user ${user.id}`,
    );
    this.tenderService.createDashboardCompany(companyId, user.id);
    return 'true';
  }

  @Get('dashboard/:companyId')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: DashboardTO })
  @ApiParam({ name: 'companyId', type: 'number' })
  async myDashboard(
    @PrimeUser() user: User,
    @Param('companyId') companyId: number,
  ): Promise<DashboardTO> {
    this.LOGGER.log(`MyDashboard - user: ${user.id} companyId: ${companyId}`);
    return this.tenderService.generateInfoDashboard(user, companyId);
  }

  @Get('history-tender')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [HistoryTender] })
  async listHistoryTender(@PrimeUser() user: User): Promise<HistoryTender[]> {
    this.LOGGER.log(`ListHistoryTender - user: ${user.id}`);
    return await this.tenderService.getListTenderHistory(user.id);
  }

  @Get(
    'history/:page/:pageSize',
  )
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [HistoryTenderDetail] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  async getHistoryPaginated(
    @PrimeUser() user: User,
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
  ): Promise<HistoryTenderDetail[]> {
    this.LOGGER.log(`GetHistoryPaginated - user: ${user.id}, page: ${page}, pageSize: ${pageSize}`);
    return await this.tenderService.getPaginatedHistoryTenders(
        user.id,
        page,
        pageSize,
      );
  }

  @Get(
    ':page/:pageSize/:companyId/:searchType/:isFavorite/:subdivisionId/:monthRequest',
  )
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [Tender] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiParam({ name: 'companyId', type: 'number' })
  @ApiParam({ name: 'searchType', type: 'string' })
  @ApiParam({ name: 'isFavorite', type: 'boolean' })
  @ApiParam({ name: 'subdivisionId', type: 'number' })
  async getPaginatedTenders(
    @PrimeUser() user: User,
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
    @Param('companyId') companyId: number,
    @Param('searchType') searchType: string,
    @Param('isFavorite') isFavorite: boolean,
    @Param('subdivisionId') subdivisionId: number,
    @Param('monthRequest') monthRequest: string,
  ): Promise<Tender[]> {
    this.LOGGER
      .log(`GetPaginatedTenders - user: ${user.id}, page: ${page}, pageSize: ${pageSize}, companyId: ${companyId} searchType: ${searchType} isFavorite: ${isFavorite} 
      subdivisionId: ${subdivisionId} monthRequest: ${monthRequest}`);
    return await this.tenderService.getPaginatedTenders(
      user.id,
      companyId,
      page,
      pageSize,
      searchType,
      subdivisionId,
      monthRequest,
    );
  }

  @Get(':code')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: Tender })
  @ApiParam({ name: 'code', type: 'string' })
  async findByCode(
    @PrimeUser() user: User,
    @Param('code') code: string,
  ): Promise<Tender | null | undefined> {
    this.LOGGER.log(`FindByCode - code: ${code} user: ${user.id}`);
    return await this.tenderService.getTenderByCode(code, user.id);
  }

  @Put('update-favorite-user-tender/:tenderId/:companyId/:isFavorite')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => String })
  @ApiParam({ name: 'tenderId', type: 'number' })
  @ApiParam({ name: 'isFavorite', type: 'string' })
  @ApiParam({ name: 'companyId', type: 'number' })
  async updateFavoriteUserCompanytender(
    @PrimeUser() user: User,
    @Param('tenderId') tenderId: number,
    @Param('companyId') companyId: number,
    @Param('isFavorite') isFavorite: string,
  ): Promise<String> {
    this.LOGGER.log(
      `Favorite - user: ${user.id}, tenderId: ${tenderId}, isFavorite: ${isFavorite} companyId: ${companyId}`,
    );
    this.tenderService.updateFavoriteUserCompanytender(
      user.id,
      companyId,
      tenderId,
      isFavorite === 'true',
    );
    return 'true';
  }

  @Delete('tender-company-user/:companyId/:tenderId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  async logicalRemoveCompanyTenderUser(
    @PrimeUser() user: User,
    @Param('companyId') companyId: number,
    @Param('tenderId') tenderId: number,
  ): Promise<String> {
    this.LOGGER.warn(
      `delete companyId ${companyId} user ${user.id} tenderId ${tenderId}`,
    );
    await this.tenderService.logicalRemoveCompanyTenderUser(
      user.id,
      companyId,
      tenderId,
    );
    return 'true';
  }

  

  @Put('add-tender-by-code-to-company/:code/:companyId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => String })
  @ApiParam({ name: 'code', type: 'string' })
  @ApiParam({ name: 'companyId', type: 'number' })
  async addTenderByCodeToCompany(
    @Param('companyId') companyId: number,
    @Param('code') code: string,
  ): Promise<String> {
    this.LOGGER.warn(
      `addTenderByCodeToCompany companyId ${companyId} code ${code}`,
    );
    return await this.tenderService.addTenderByCodeToCompany(code, companyId);
  }
}
