import { Injectable } from '@nestjs/common';
import { and, eq, isNotNull, isNull, sql } from 'drizzle-orm';
import { PrimeLogger, schema, TxType } from 'src/framework';
import { KeywordRepository } from 'src/licitaapp/application';
import { DBConfigService } from 'src/framework/infrastructure/drizzle';
import { Keyword, InsertKeyword, Metadata } from 'src/licitaapp/domain';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class KeywordRepositoryImpl implements KeywordRepository {
  private readonly LOGGER = new PrimeLogger(KeywordRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async getAllWithouthMetadata(tx?: TxType): Promise<Keyword[]> {
    this.LOGGER.log(`getAllWithouthMetadata`);
    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
      })
      .from(schema.keywordTable)
      .where(and(eq(schema.keywordTable.active, true), isNull(schema.keywordTable.metadata)))
      .then((rows) => {
        return rows.map((row) => new Keyword(row.id, row.value, row.origin));
      }
    );
  }
  async findByCompanyId(companyId: number, isNullMetadata: boolean, tx?: TxType): Promise<Keyword[]> {
    const baseWhereClause = and(
      eq(schema.keywordTable.companyId, companyId),
      eq(schema.keywordTable.active, true)
    );

    const metadataClause = isNullMetadata  && isNotNull(schema.keywordTable.metadata);

    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
        metadata: schema.keywordTable.metadata,
      })
      .from(schema.keywordTable)
      .where(metadataClause ? and(baseWhereClause, metadataClause): and(baseWhereClause))
      .then((rows) => {
        return rows.map((row) => new Keyword(row.id, row.value, row.origin, undefined, undefined, row.metadata));
      });
  }

  async save(keyword: InsertKeyword, userId: number, tx?: TxType): Promise<Keyword> {
    const validatedKeyword = schema.keywordTableInsertSchema.parse({
      value: keyword.value,
      origin: keyword.origin,
      active: true,
      userId: userId,
      companyId: keyword.companyId,
    });

    await (tx || this.db.conn).execute(
        sql.raw(`INSERT INTO keyword (active, value, origin, company_id, user_id) VALUES 
          (true, ${keyword.value}, ${keyword.origin}, ${keyword.companyId},${userId});`)            
      )
      .catch((err) => {
        this.LOGGER.error(`Error fetching registered users: ${err}`);
        throw new Error(`Error fetching registered users: ${err}`);
      });
    return new Keyword(0, validatedKeyword.value, validatedKeyword.origin, undefined, undefined, undefined);
  }

  async saveAll(keywords: InsertKeyword[], userId: number, tx?: TxType): Promise<Keyword[]> {
    /*const validatedKeywords = keywords.map((keyword) =>
      schema.keywordTableInsertSchema.parse({
        value: keyword.value,
        origin: keyword.origin,
        active: true,
        userId: userId,
        metadata: keyword.metadata as Metadata ?? undefined,
        companyId: keyword.companyId,
      }),
    );
    const insertedKeywordIds = await (tx || this.db.conn)
      .insert(schema.keywordTable)
      .values(validatedKeywords)
      .$returningId()
      .then((rows) => {
        return rows.map((row) => row.id);
      });

    return await Promise.all(
      insertedKeywordIds.map((id) => this.findById(id)),
    ).then((keywords) => {
      return keywords.filter((keyword) => keyword !== null) as Keyword[];
    });*/
    return [];
  }

  async updateMetadata(keywordId: number, metadata: Metadata): Promise<void> {
    this.LOGGER.log(`updateMetadata keyword: ${keywordId}`);
    
    await (this.db.conn)
      .update(schema.keywordTable)
      .set({ 
        updatedAt: TenderUtil.getCurrentSystemDate(), 
        metadata: metadata, 
      })
      .where(eq(schema.keywordTable.id, keywordId));

  }

  async findById(id: number, tx?: TxType): Promise<Keyword | null | undefined> {
    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
      })
      .from(schema.keywordTable)
      .where(eq(schema.keywordTable.id, id))
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }

        return new Keyword(rows[0].id, rows[0].value, rows[0].origin);
      });
  }

  async logicalRemove(id: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`logicalRemove keyword: ${id}`);
    await (tx || this.db.conn)
      .update(schema.keywordTable)
      .set({ active: false, deletedAt: TenderUtil.getCurrentSystemDate() })
      .where(eq(schema.keywordTable.id, id));
  }

  async deleteByCompanyId(companyId: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`deleteByCompanyId keyword: ${companyId}`);
    await (tx || this.db.conn)
      .delete(schema.keywordTable)
      .where(eq(schema.keywordTable.companyId, companyId));
  }
}
