import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { and, count, desc, eq, inArray, isNull } from 'drizzle-orm';
import { PrimeLogger, schema } from 'src/framework';
import { DBConfigService, TxType } from 'src/framework/infrastructure/drizzle';
import { TenderRepository, Licitacion } from 'src/licitaapp';
import { InsertTender, Metadata, RequestTender, Tender, TenderHistoryTO, TenderTO } from 'src/licitaapp/domain';
import { TenderRow } from 'src/licitaapp/domain/type/tender.types';
import { TenderUtil } from 'src/licitaapp/domain/util';


@Injectable()
export class TenderRepositoryImpl implements TenderRepository {
    private readonly LOGGER = new PrimeLogger(TenderRepositoryImpl.name);
    private readonly limitGeoTendersDashboard: number;
    constructor(private readonly db: DBConfigService,
      private readonly configService: ConfigService,
    ) {
      this.limitGeoTendersDashboard = this.configService.get<number>(
        'LIMIT_GEO_TENDERS_DASHBOARD',
        5
      );
    }
  async findInfohistoryTender(tenderId: number[]): Promise<TenderHistoryTO[]> {
    this.LOGGER.log(`findInfohistoryTender Tender IDs: ${tenderId.filter(id => id != 0)}`);
    if(tenderId.length === 0){
      return [];
    }
    const selectFields = {
      tenderId: schema.tenderTable.id,
      code: schema.tenderTable.code,
      name: schema.tenderTable.name,
      description: schema.tenderTable.description,
      details: schema.tenderTable.details,
      createdAt: schema.tenderTable.createdAt,
      updatedAt: schema.tenderTable.updatedAt,
    };

    return await (this.db.conn)
      .select(selectFields)
      .from(schema.tenderTable)
      .where(inArray(schema.tenderTable.id, tenderId.filter(id => id != 0)))
      .then((rows: any[]) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map((row) => {
          return {
            tenderId: row.tenderId,
            code: row.code,
            name: row.name,
            description: row.description,
            details: row.details,
            createdAt: row.createdAt,
            updatedAt: row.updatedAt,
          } as TenderHistoryTO;
        });
      });
  }
  async erraseByListId(tenderIds: number[]): Promise<void> {
    this.LOGGER.log(`erraseByListId Tender IDs: ${tenderIds}`);
    await this.db.conn
      .delete(schema.tenderTable)
      .where(inArray(schema.tenderTable.id, tenderIds)).execute();
  }
    async findByCode(code: string): Promise<Tender | undefined> {
    return await this.selectTenderQuery()
      .where(and(eq(schema.tenderTable.code, code), eq(schema.tenderTable.active, true)))
      .then((rows: TenderRow[]) => {
        if (rows.length === 0) {
          return undefined;
        }

        return this.mapRowToTender(rows[0]);
      });
  }

  async logicalRemove(tenderId: number): Promise<void> {
    this.LOGGER.log(`logicalRemove Tender: ${tenderId}`);
    await (this.db.conn)
      .update(schema.tenderTable)
      .set({ 
        active: false,
        deletedAt: TenderUtil.getCurrentSystemDate(),
      })
      .where(eq(schema.tenderTable.id, tenderId));
  }

  async getAllWithouthMetadata(tx?: TxType) {
    this.LOGGER.log(`getAllWithouthMetadata`);
    return await (tx || this.db.conn)
      .select({
        id: schema.tenderTable.id,
        description: schema.tenderTable.description,
        detail: schema.tenderTable.details,
      })
      .from(schema.tenderTable)
      .where(and(eq(schema.tenderTable.active, true), isNull(schema.tenderTable.metadata)))
      .then((rows) => {
        return rows.map((row) => ({
          id: row.id,
          description: row.description,
          detail: row.detail
        }));
      }
    );
  }

  async updateMetadata(tenderId: number, metadata: Metadata){
    this.LOGGER.log(`updateMetadata Tender: ${tenderId}`);    
    await (this.db.conn)
      .update(schema.tenderTable)
      .set({ 
        updatedAt: TenderUtil.getCurrentSystemDate(), 
        metadata: metadata, 
      })
      .where(eq(schema.tenderTable.id, tenderId));
  }

  async findByDates(initDate: Date, endDate: Date ): Promise<RequestTender[]> {
    this.LOGGER.log(`findByDates - initDate: ${initDate} endDate: ${endDate}`);

    var listTender: RequestTender[] = [];
    await this.db.conn.execute(
      `SELECT id, code, name, description, metadata FROM tender WHERE created_at BETWEEN '${initDate.toISOString().slice(0, 19).replace("T", " ")}' AND '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true AND metadata IS NOT NULL`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        const newTender = new RequestTender();
        newTender.id = item.id;
        newTender.code = item.code;
        newTender.name = item.name;
        newTender.description = item.description;
        newTender.metadata = item.metadata;
        return newTender;
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByDates: ${err}`);
    });
    
    return listTender;
  }

  async findWithoutCloseDate(): Promise<Tender[]> {
    this.LOGGER.log(`findWithoutCloseDate`);
    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE close_date IS NULL`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findWithoutCloseDate: ${err}`);
    });
    
    return listTender;
  }

  async findWithoutSubdivision(): Promise<Tender[]> {
    this.LOGGER.log(`findWithoutCity`);
    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE subdivision_id IS NULL`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findWithoutCity: ${err}`);
    });
    
    return listTender;
  }

  async findByCloseDate(endDate: Date ): Promise<Tender[]> {
    this.LOGGER.log(`findByCloaseDate - endDate: ${endDate}`);

    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE close_date <= '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
      return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByCloseDate: ${err}`);
    });
    
    return listTender;
  }

  async findByDatesUpset(initDate: Date, endDate: Date ): Promise<Tender[]> {
    this.LOGGER.log(`findByDates - initDate: ${initDate.toISOString().slice(0, 19).replace("T", " ")} endDate: ${endDate.toISOString().slice(0, 19).replace("T", " ")}`);

    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE created_at BETWEEN '${initDate.toISOString().slice(0, 19).replace("T", " ")}' AND '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true OR 
      updated_at BETWEEN '${initDate.toISOString().slice(0, 19).replace("T", " ")}' AND '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByDates: ${err}`);
    });
    
    return listTender;
  }

  async findById(id: number): Promise<Tender | undefined> {
    return await this.selectTenderQuery()
      .where(and(eq(schema.tenderTable.id, id), eq(schema.tenderTable.active, true)))
      .then((rows: TenderRow[]) => {
        if (rows.length === 0) {
          return undefined;
        }

        return this.mapRowToTender(rows[0]);
      });
  }

  /**
   * Si estan activas significa que aun estan en curso.
   * @param subdivisionId 
   * @returns 
   */
  async findBySubdivisionId(subdivisionId: number, withLimit: boolean = false): Promise<Tender[]> {
    try {
      const query = this.selectTenderQuery()
        .where(and(eq(schema.tenderTable.subdivisionId, subdivisionId), eq(schema.tenderTable.active, true)));

      if (withLimit) {
        query.limit(Number(this.limitGeoTendersDashboard));
      }
      query.orderBy(desc(schema.tenderTable.createdAt));
      const rows = await query;
      return rows.map((item: any) => this.mapRowToTender(item));
    } catch (err) {
      this.LOGGER.error(`Error findBySubdivisionId: ${err}`);
      return [];
    }
  }

  async upsert(tender: InsertTender): Promise<number> {
    const validatedTender = schema.tenderTableInsertSchema.parse({
      code: tender.code,
      name: tender.name,
      description: tender.description,
      details: tender.details,
      subdivisionId: tender.subdivisionId,
    });

    const insertedTenderId = await this.db.conn
      .insert(schema.tenderTable)
      .values({
        code: validatedTender.code,
        name: validatedTender.name,
        description: validatedTender.description,
        subdivisionId: validatedTender.subdivisionId,
        details: tender.details,
      })
      .onDuplicateKeyUpdate({
        set: {
          code: validatedTender.code,
          name: validatedTender.name,
          description: validatedTender.description,
          subdivisionId: validatedTender.subdivisionId,
          details: tender.details,
          updatedAt: TenderUtil.getCurrentSystemDate(),
        },
      })
      .$returningId()
      .then((rows) => {
        return rows[0].id;
      });

    return insertedTenderId;
  }

  async countActiveTenders(): Promise<number> {
    this.LOGGER.log(`countActiveTenders`);
      return await this.db.conn
        .select({count: count()})
        .from(schema.tenderTable)
        .where(
          and(
            eq(schema.tenderTable.active, true),
          ),
        )
        .then((rows) => {
          return rows[0].count;
        });
  }

  async getLogicalRemoveTenderIds(): Promise<TenderTO[]> {
    this.LOGGER.log(`getLogicalRemoveTenderIds`);
    return await this.db.conn
      .select({ id: schema.tenderTable.id, code: schema.tenderTable.code, details: schema.tenderTable.details, description: schema.tenderTable.description })
      .from(schema.tenderTable)
      .where(eq(schema.tenderTable.active, false))
      .then((rows) => {
        return rows.map((row) => new TenderTO(row.id, row.code, row.details, row.description));
      });
  }

  async updateField(tender: {
    id: number;
    active: boolean;
    detail?: Licitacion;
    subdivisionId?: number | null;
  }): Promise<void> {
    
    await this.db.conn
      .update(schema.tenderTable)
      .set({
        active: tender.active,
        subdivisionId: tender.subdivisionId,
        updatedAt: TenderUtil.getCurrentSystemDate(),
        details: tender.detail,
        deletedAt: tender.active ? null : TenderUtil.getCurrentSystemDate(),
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
  }

  async updateCloseDate(tender: {
    id: number;
    closeDate: Date | null;
  }): Promise<void> {
    
    await this.db.conn
      .update(schema.tenderTable)
      .set({
        closeDate: tender.closeDate,
        updatedAt: TenderUtil.getCurrentSystemDate()
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
  }

  async updateLogicalRemove(tender: {
    id: number;
    state: boolean;
  }): Promise<void> {
    
    if(!tender.state){
      await this.db.conn
      .update(schema.tenderTable)
      .set({
        active: tender.state,
        deletedAt: TenderUtil.getCurrentSystemDate()
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
    }
  }

  async updateSubdivision(tender: {
    id: number;
    subdivisionId?: number | null;
  }): Promise<void> {
    
    await this.db.conn
      .update(schema.tenderTable)
      .set({
        subdivisionId: tender.subdivisionId,
        updatedAt: TenderUtil.getCurrentSystemDate()
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
  }
  
  private selectTenderQuery() {
    return this.db.conn
      .select({
        id: schema.tenderTable.id,
        code: schema.tenderTable.code,
        name: schema.tenderTable.name,
        description: schema.tenderTable.description,
        details: schema.tenderTable.details,
        createdAt: schema.tenderTable.createdAt,
        updatedAt: schema.tenderTable.updatedAt,
        closeDate: schema.tenderTable.closeDate,
      })
      .from(schema.tenderTable)
      .$dynamic();
  }
  
  private mapRowToTender(row: TenderRow): Tender {
    return new Tender(
      row.id,
      row.code,
      row.name,
      row.description,
      row.details,
      row.createdAt,
      row.updatedAt,
      row.closeDate,
      false,
      row.subdivisionId ? row.subdivisionId : undefined,
    );
  }
}
