import { Injectable } from '@nestjs/common';
import { and, asc, count, eq, inArray, like } from 'drizzle-orm';
import { PrimeLogger, schema } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle';
import { UserCompanyTenderRepository } from 'src/licitaapp/application/repository/user-company-tender-repository/user-company-tender-repository.interface';
import { HistoryTenderDetail, Tender, TenderTO } from 'src/licitaapp/domain';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class UserCompanyTenderRepositoryImpl
  implements UserCompanyTenderRepository
{
  private readonly LOGGER = new PrimeLogger(
    UserCompanyTenderRepositoryImpl.name,
  );
  constructor(private readonly db: DBConfigService) {}
  async erraseUserCompanyTender(tenderIds: TenderTO[]): Promise<void> {
    this.LOGGER.log(`erraseUserCompanyTender - tenderIds: ${tenderIds.length}`);
    await this.db.conn
      .delete(schema.userCompanyTenderTable)
      .where(inArray(schema.userCompanyTenderTable.tenderId, tenderIds.map(t => t.tenderId)))
      .execute();
  }

  async saveAll(
    userId: number,
    companyId: number,
    tenderIds: number[],
    source: string,
  ): Promise<boolean> {
    try {
      await this.db.conn.transaction(async (tx) => {
        for (const tenderId of tenderIds) {
          await tx
            .insert(schema.userCompanyTenderTable)
            .values({ userId, companyId, tenderId, source })
            .onDuplicateKeyUpdate({
              set: {
                userId,
                companyId,
                tenderId,
                source,
                updatedAt: TenderUtil.getCurrentSystemDate(),
              },
            })
            .execute();
        }
      });
      return true;
    } catch (error) {
      throw new Error(error);
    }
  }

  async updateFavoriteUserCompanyTender(
    userId: number,
    companyId: number,
    tenderId: number,
    isFavorite: boolean,
  ) {
    this.LOGGER.log(
      `updateFavoriteUserCompanyTender userId: ${userId} companyId: ${companyId} tenderId: ${tenderId} isFavorite: ${isFavorite}`,
    );
    await this.db.conn
      .update(schema.userCompanyTenderTable)
      .set({ isFavorite })
      .where(
        and(
          eq(schema.userCompanyTenderTable.userId, userId),
          eq(schema.userCompanyTenderTable.companyId, companyId),
          eq(schema.userCompanyTenderTable.tenderId, tenderId),
          eq(schema.userCompanyTenderTable.active, true),
          eq(schema.userCompanyTenderTable.isFavorite, !isFavorite),
        ),
      )
      .execute();
  }

  async save(
    userId: number,
    companyId: number,
    tenderId: number,
    source: string,
    matchResult: string,
  ): Promise<boolean> {
    try {
      await this.db.conn.transaction(async (tx) => {
        await tx
          .insert(schema.userCompanyTenderTable)
          .values({ userId, companyId, tenderId, source, matchResult })
          .onDuplicateKeyUpdate({
            set: {
              userId,
              companyId,
              tenderId,
              source,
              matchResult,
              updatedAt: TenderUtil.getCurrentSystemDate(),
            },
          })
          .execute();
      });
      return true;
    } catch (error) {
      throw new Error(error);
    }
  }

  async findAllUserIdWithTenders() {
    return await this.db.conn
      .select({
        userId: schema.userCompanyTenderTable.userId,
      })
      .from(schema.userCompanyTenderTable)
      .innerJoin(
        schema.userTable,
        eq(schema.userCompanyTenderTable.userId, schema.userTable.id),
      )
      .where(eq(schema.userTable.active, true))
      .groupBy(schema.userCompanyTenderTable.userId)
      .then((rows) => {
        return rows.map((row) => {
          return row.userId;
        });
      });
  }

  async findTenderByUserCompanyId(
    userId: number,
    companyId: number,
  ): Promise<Tender[]> {
    this.LOGGER.log(
      `findTenderByUserCompanyId userId: ${userId} companyId: ${companyId}`,
    );
    return await this.db.conn
      .select()
      .from(schema.tenderTable)
      .innerJoin(
        schema.userCompanyTenderTable,
        eq(schema.userCompanyTenderTable.tenderId, schema.tenderTable.id),
      )
      .where(
        and(
          eq(schema.userCompanyTenderTable.userId, userId),
          eq(schema.userCompanyTenderTable.companyId, companyId),
          eq(schema.userCompanyTenderTable.active, true),
          eq(schema.tenderTable.active, true),
        ),
      )
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) =>
            new Tender(
              row.tender.id,
              row.tender.code,
              row.tender.name,
              row.tender.description,
              row.tender.details,
              row.tender.createdAt,
              row.tender.updatedAt,
              row.tender.closeDate,
              row.user_company_tender.isFavorite,
              row.tender.subdivisionId,
              row.user_company_tender.source,
            ),
        );
      });
  }

  async getPaginatedTenders(
    userId: number,
    companyId: number,
    page: number,
    pageSize: number,
    searchType: string,
    subdivisionId: number,
    monthRequest: string,
  ): Promise<Tender[]> {
    const offset = (page - 1) * pageSize;

    if (searchType === 'NORMAL_TENDERS' && monthRequest.length > 5) {
      const { start, end } = TenderUtil.getMonthStartAndEnd(
        new Date(monthRequest),
      );
      await this.db.conn
        .execute(
          `select tender.id, tender.code, tender.name, tender.description, tender.details, tender.created_at, tender.updated_at, user_company_tender.is_favorite
          from user_company_tender 
          LEFT join tender on tender.id = user_company_tender.tender_id
          where (user_company_tender.user_id = ${userId} and 
          user_company_tender.company_id = ${+companyId} and 
          user_company_tender.is_favorite = false and 
          user_company_tender.source like 'Coincidencias %' and 
          tender.created_at between '${start.toISOString().slice(0, 19).replace('T', ' ')}' and '${end.toISOString().slice(0, 19).replace('T', ' ')}' and 
          user_company_tender.active = true and tender.active = true and tender.subdivision_id = ${+subdivisionId}) order by tender.close_date desc limit ${pageSize} OFFSET ${offset}`,
        )
        .then((rows: any) => {
          if (rows.length === 0) {
            return [];
          }
          rows.map((row: any) => {
            return new Tender(
              row.id,
              row.code,
              row.name,
              row.description,
              row.details,
              row.created_at,
              row.updated_at,
              null,
              row.is_favorite,
            );
          });
        })
        .catch((err) => {
          this.LOGGER.error(`Error findByDates: ${err}`);
        });
    }

    let whereSentence = and(eq(schema.tenderTable.active, true));

    /*if (searchType === 'AGILE_TENDERS') {
      whereSentence = and(
        eq(schema.userCompanyTenderTable.userId, userId),
        //eq(schema.userCompanyTenderTable.isBuyAgile, true),
        eq(schema.userCompanyTenderTable.companyId, companyId),
        eq(schema.tenderTable.active, true),
      );
    } else*/ if (searchType === 'NORMAL_TENDERS') {
      if (subdivisionId != 0) {
        whereSentence = and(
          eq(schema.userCompanyTenderTable.userId, userId),
          eq(schema.userCompanyTenderTable.isFavorite, false),
          eq(schema.userCompanyTenderTable.companyId, +companyId),
          like(schema.userCompanyTenderTable.source, 'Cerca en %'),
          eq(schema.userCompanyTenderTable.active, true),
          eq(schema.tenderTable.active, true),
          eq(schema.tenderTable.subdivisionId, +subdivisionId),
        );
      } else {
        whereSentence = and(
          eq(schema.userCompanyTenderTable.userId, userId),
          eq(schema.userCompanyTenderTable.active, true),
          eq(schema.userCompanyTenderTable.isFavorite, false),
          //eq(schema.userCompanyTenderTable.isBuyAgile, false),
          eq(schema.tenderTable.active, true),
          eq(schema.userCompanyTenderTable.companyId, companyId),
        );
      }
    } else if (searchType === 'FAVORITE_TENDERS') {
      whereSentence = and(
        eq(schema.userCompanyTenderTable.userId, userId),
        eq(schema.userCompanyTenderTable.companyId, companyId),
        eq(schema.userCompanyTenderTable.active, true),
        eq(schema.userCompanyTenderTable.isFavorite, true),
        eq(schema.tenderTable.active, true),
      );
    }

    const selectFields = {
      field1: schema.tenderTable.id,
      field2: schema.tenderTable.code,
      field3: schema.tenderTable.name,
      field4: schema.tenderTable.description,
      field5: schema.tenderTable.details,
      field6: schema.tenderTable.createdAt,
      field7: schema.tenderTable.updatedAt,
      field8: schema.userCompanyTenderTable.isFavorite,
    };

    const query = this.db.conn
      .select(selectFields)
      .from(schema.tenderTable)
      .where(whereSentence)
      .limit(Number(pageSize))
      .offset(offset);

    //if (searchType !== 'ALL_TENDERS') {
    query.innerJoin(
      schema.userCompanyTenderTable,
      eq(schema.userCompanyTenderTable.tenderId, schema.tenderTable.id),
    );
    query.orderBy(asc(schema.tenderTable.closeDate));
    /*}else{
      query.orderBy((desc(schema.tenderTable.updatedAt)));
    }*/

    const rows = await query;

    if (rows.length === 0) {
      return [];
    }

    return rows.map(
      (row) =>
        new Tender(
          row.field1,
          row.field2,
          row.field3,
          row.field4,
          row.field5,
          row.field6,
          row.field7,
          null,
          row.field8,
        ),
    );
  }

  async getHistoryTendersFavorite(userId: number, dateRequest: Date) {
    this.LOGGER.log(
      `getHistoryTendersFavorite userId: ${userId} dateRequest: ${dateRequest}`,
    );
    const { start, end } = TenderUtil.getDayStartAndEnd(dateRequest);
    var listTender: Tender[] = [];
    await this.db.conn
      .execute(
        `select tender.id, tender.code, tender.name, tender.description, tender.details, tender.created_at, tender.updated_at, user_company_tender.is_favorite from tender inner join 
          user_company_tender on user_company_tender.tender_id = tender.id where (user_company_tender.user_id = ${userId}
           and tender.close_date between '${start.toISOString().slice(0, 19).replace('T', ' ')}' and '${end.toISOString().slice(0, 19).replace('T', ' ')}' and 
           user_company_tender.active = true and tender.active = true and user_company_tender.is_favorite = true)`,
      )
      .then((rows: any) => {
        const items = rows[0];
        listTender = items.map((row: any) => {
          const outObj = new Tender(
            row.id,
            row.code,
            row.name,
            row.description,
            row.details,
            row.created_at,
            row.updated_at,
            null,
            row.is_favorite,
          );
          outObj.isErrased = false;
          return outObj;
        });
      })
      .catch((err) => {
        this.LOGGER.error(`Error findWithoutCity: ${err}`);
      });

    return listTender;
  }

  async getPaginatedHistoryTenders(
    userId: number,
    page: number,
    pageSize: number,
  ): Promise<HistoryTenderDetail[]> {
    this.LOGGER.log(`getPaginatedHistoryTenders userId: ${userId}, page: ${page}, pageSize: ${pageSize}`);
    const offset = (page - 1) * pageSize;
    const selectFields = {
      id: schema.userHistoryTenderTable.id,
      tenderId: schema.userHistoryTenderTable.tenderId,
      codeTender: schema.userHistoryTenderTable.codeTender,
      metadata: schema.userHistoryTenderTable.metadata,
    };
    
    let whereSentence = and(eq(schema.userHistoryTenderTable.userId, userId), 
        eq(schema.userHistoryTenderTable.active, true));
        
    const query = this.db.conn
      .select(selectFields)
      .from(schema.userHistoryTenderTable)
      .where(whereSentence)
      .limit(Number(pageSize))
      .offset(offset);

    const rows = await query;

    if (rows.length === 0) {
      return [];
    }

    return rows.map(
      (row) =>{
        const metadataOldTender = row.metadata;
        const out = new HistoryTenderDetail();
        out.isErrased = metadataOldTender ? true : false;
        out.code = row.codeTender ? row.codeTender : '';
        out.id = row.tenderId ? row.tenderId : 0;
        if(metadataOldTender){
          out.labelAmount = metadataOldTender.labelAmount;
          out.typeOfMoney =metadataOldTender.typeOfMoney;
          out.labelLastUpdated =metadataOldTender.labelLastUpdated;
          out.labelCloseTender = metadataOldTender.labelCloseTender;
          out.description = metadataOldTender.description;
          out.name = metadataOldTender.name;
        }
        return out;
      }
        
    );
  }

  async logicalRemoveCompanyTenderUser(
    userId: number,
    companyId: number,
    tenderId: number,
  ): Promise<void> {
    await this.db.conn
      .update(schema.userCompanyTenderTable)
      .set({ active: false, deletedAt: TenderUtil.getCurrentSystemDate() })
      .where(
        and(
          eq(schema.userCompanyTenderTable.userId, userId),
          eq(schema.userCompanyTenderTable.companyId, companyId),
          eq(schema.userCompanyTenderTable.tenderId, tenderId),
        ),
      )
      .execute();
  }

  async erraseJoinCompanyTender(companyId: number) {
    this.LOGGER.log(`erraseJoinCompanyTender - companyId: ${companyId}`);
    await this.db.conn
      .delete(schema.userCompanyTenderTable)
      .where(
        and(
          eq(schema.userCompanyTenderTable.companyId, companyId),
          eq(schema.userCompanyTenderTable.isFavorite, false),
        ),
      )
      .execute();
  }

  async countActiveTendersCompany(companyId: number): Promise<number> {
    return await this.db.conn
      .select({ count: count() })
      .from(schema.userCompanyTenderTable)
      .where(
        and(
          eq(schema.userCompanyTenderTable.companyId, companyId),
          eq(schema.userCompanyTenderTable.active, true),
        ),
      )
      .then((rows) => {
        return rows[0].count;
      });
  }

  async getUsersIdsByCompanyId(companyId: number): Promise<number[]> {
    this.LOGGER.log(`getUsersIdsByCompanyId companyId: ${companyId}`);
    return await this.db.conn
      .select({ userId: schema.userCompanyTable.userId })
      .from(schema.userCompanyTable)
      .where(eq(schema.userCompanyTable.companyId, companyId))
      .then((rows) => {
        return rows.map((row) => row.userId);
      });
  }
}
