import { Inject, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { Cron, CronExpression } from '@nestjs/schedule';
import { InsertNotificationRecord, PrimeLogger, UserCampaignsService } from 'src/framework';
import { NotificationRecordService } from 'src/framework/application/service/notification-record-service/notification-record-service.interface';
import { CompanyService, UserCompanyService } from 'src/licitaapp/application';
import { ScheduleMpService } from 'src/licitaapp/application/service/schedule-mp-service/schedule-mp-service.interface';
import { TenderFetcherService } from 'src/licitaapp/application/service/tender-fetcher-service/tender-fetcher-service.interface';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class ScheduleMpServiceImpl implements ScheduleMpService {

    private readonly LOGGER = new PrimeLogger(ScheduleMpServiceImpl.name);
    private readonly isCronActive: boolean;
    private hourToSearchTenders = [0, 4, 12, 17];
    private perFiveMinutes = [5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55];
    private allHours = [6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23];
    constructor(
        private readonly configService: ConfigService,
        @Inject('TenderService') private readonly tenderService: TenderService,
        @Inject('TenderFetcherService') private readonly tenderFetcherService: TenderFetcherService,
        @Inject('UserCampaignsService') private readonly userCampaignsService: UserCampaignsService,
        @Inject('NotificationRecordService') private readonly notificationRecordService: NotificationRecordService,
        @Inject('UserCompanyService') private readonly userCompanyService: UserCompanyService,
    ) {
        this.isCronActive = this.configService.get<string>(
        'CRON_JOB_ENABLED',
        'true'
        ).toLowerCase() ===
        'true';
    }
    
    @Cron(CronExpression.EVERY_MINUTE, {
       name: 'mainCentralScheduleMp',
       timeZone: TenderUtil.TIMEZONE,
    })
    async mainCentralScheduleMp() {

        if (!this.isCronActive){
            this.LOGGER.warn('mainCentralScheduleMp - Cron job is disabled');
            return;
        }
        
        const currentTime = TenderUtil.getCurrentSystemDate();
        this.LOGGER.log(
            `mainCentralScheduleMp - currentTime: ${currentTime.toLocaleString('es-CL', { timeZone: TenderUtil.TIMEZONE })}`
        );
        // task per minute
        this.tenderService.tenderTaskPerMinutes();

        const hour = currentTime.getHours();
        const minute = currentTime.getMinutes();
        this.LOGGER.log(
            `mainCentralScheduleMp - hour: ${hour}, minute: ${minute}`
        );
        
        // todas las 0, 4, 12, 17 horas
        if (this.hourToSearchTenders.includes(hour) && minute === 0) {
            this.tenderFetcherService.fetchTendersOfTheDay();
        }
        
        // task per 5 minutes
        if (this.perFiveMinutes.includes(minute)) {
            this.tenderService.tenderTaskEvery5Minutes();
        }

        // 12:22 de la mañana se ejecuta una vez al día
        if (hour === 0 && minute === 22) {
            this.LOGGER.log('mainCentralScheduleMp - Executing daily tasks');
            this.tenderService.tenderTaskPerDay();
            this.tenderService.erraseOldTenders();
        }

        // cada 2 horas en horas impares con 7 minutos
        if(hour % 2 !== 0 && minute === 7) {
            this.LOGGER.log('mainCentralScheduleMp - Executing tasks every 2 hours');
            this.tenderService.tenderTaskBy2Hour();
        }

        // cada hora al minuto 17 (entre las 6 y las 23)
        if( this.allHours.includes(hour) && minute === 17) {
            this.LOGGER.log('mainCentralScheduleMp - Executing tasks every hour');
            this.tenderService.tenderTaskByHour();
        }

        this.LOGGER.log(`mainCentralScheduleMp - Running CAMPAIGN at ${hour}:${minute}`);
        this.createCampaign(hour, minute);
    }

    
    async createCampaign(hour: number, minute: number): Promise<void> {
        const campaings = await this.userCampaignsService.getActiveCampaigns();

        if (!campaings || campaings.length === 0) {
            this.LOGGER.warn('createCampaign - No active campaigns found');
            return;
        }
        this.LOGGER.log(`createCampaign - Found ${campaings.length} active campaigns`);
        for (const campaign of campaings) {
            const daysOfWeek = campaign.dayOfWeek.split('-').map(Number);
            if(!daysOfWeek.includes(TenderUtil.getCurrentSystemDate().getDay())) {
                this.LOGGER.log(`createCampaign - Skipping campaign ${campaign.id} for day ${campaign.dayOfWeek}`);
                continue;
            }
            this.LOGGER.log(`createCampaign - Processing campaign ${campaign.id}`);
            if( !campaign.defHour || !campaign.defMinute) {
                this.LOGGER.warn(`createCampaign - Campaign ${campaign.id} does not have a valid time set`);
                continue;
            }else{
                this.LOGGER.warn(`Campaign ${campaign.id} has time set to ${campaign.defHour}:${campaign.defMinute} compare with ${hour}:${minute}`);
                if(campaign.defHour == hour && campaign.defMinute == minute) {
                    this.LOGGER.log(`createCampaign - Creating campaign for user with id: ${campaign.id}`);
                    const activeUserIds = await this.userCompanyService.getUserIdsWithActiveCompany();
                    const notificationsArray: InsertNotificationRecord[] = [];
                    for(const userID of activeUserIds){
                        this.LOGGER.log(`createCampaign - Creating notification for user with id: ${userID}`);
                        notificationsArray.push({
                            userId: userID,
                            title: campaign.title,
                            defaultMessage: campaign.defaultMessage,
                            active: true,
                        });                
                    }        
                    notificationsArray.length > 0 && await this.notificationRecordService.saveAll(notificationsArray);
                }
            }
        }

    }

}
