import { Inject, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as moment from 'moment-timezone';
import { PrimeLogger } from 'src/framework';
import { LicitacionesQueryParams, LicitacionesResponse } from 'src/licitaapp/application';
import { MercadoPublicoRepository } from 'src/licitaapp/application/mercado-publico/mp.repository';
import { ApplicationLogService } from 'src/licitaapp/application/service/application-log-service/application-log-service.interface';
import { TenderFetcherService } from 'src/licitaapp/application/service/tender-fetcher-service/tender-fetcher-service.interface';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { ApplicationTypeEnum } from 'src/licitaapp/domain/enum/enum.definition';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { inspect } from 'util';

@Injectable()
export class TenderFetcherServiceImpl implements TenderFetcherService {
    private readonly LOGGER = new PrimeLogger(TenderFetcherServiceImpl.name);
    static readonly TIMEZONE = 'America/Santiago';
    private readonly maxRetries;
    private readonly retryInterval;
    private readonly fetchFromDaysAgo;
    private readonly enabled;

    constructor(
    private readonly configService: ConfigService,
    @Inject('MercadoPublicoRepository') private readonly mercadoPublicoAPI: MercadoPublicoRepository,
    @Inject('TenderService') private readonly tenderService: TenderService,
    @Inject('ApplicationLogService') private readonly applicationLogService: ApplicationLogService,
  ) {
    this.maxRetries = this.configService.get<number>(
      'TENDER_FETCHER_MAX_RETRIES',
      10,
    );
    this.retryInterval = this.configService.get<number>(
      'TENDER_FETCHER_RETRY_INTERVAL',
      1000,
    );
    this.fetchFromDaysAgo = this.configService.get<number>(
      'TENDER_FETCHER_FETCH_FROM_DAYS_AGO',
      90,
    );
    this.enabled =
      this.configService.get<'true' | 'false'>(
        'TENDER_FETCHER_ENABLED',
        'false',
      ) === 'true';
  }

    async fetchTendersWithRetry(args: LicitacionesQueryParams): Promise<LicitacionesResponse | undefined> {
        const delay = (ms: number) =>
        new Promise((resolve) => setTimeout(resolve, ms));

        for (let attempt = 1; attempt <= this.maxRetries; attempt++) {
            try {
                return await this.mercadoPublicoAPI.licitaciones(args);
            } catch (error) {
                if (attempt === Number(this.maxRetries)) {
                throw error;
            }
            await delay(attempt * this.retryInterval); // Exponential backoff
            }
        }
    }
    async processTenders(args: LicitacionesQueryParams): Promise<boolean> {
        this.LOGGER.debug(
      `Fetching tenders with args: ${JSON.stringify(args)}`
    );

    const startTime = Date.now();
    const applicationLogId = await this.applicationLogService.save({
      userName: 'SYSTEM',
      statusTypeId: 8,
      detail: 'Iniciado a las ' + TenderUtil.getCurrentSystemDate().toLocaleTimeString(),
      type: ApplicationTypeEnum.SEARCH_TENDERS_MERCADO_PUBLICO_LOG,
    });

    try {
      const tenders = await this.fetchTendersWithRetry(args);
      if (tenders && tenders.Cantidad > 0) {
        let processed = 0;
        this.LOGGER.debug(
          `Fetched ${tenders.Cantidad} tenders. Processing...`
        );

        for (const tender of tenders.Listado) {
          processed++;
          this.LOGGER.debug(
            `Tender: ${tender.Nombre}, codigo ${tender.CodigoExterno}`
          );
          try {
            const codeDetails = await this.fetchTendersWithRetry({
              codigo: tender.CodigoExterno,
            });

            if (codeDetails?.Listado) {
              for (const tenderDetails of codeDetails.Listado) {
                const trace = `Upserting tender: ${tenderDetails.CodigoExterno} processed ${processed}/${tenders.Cantidad} tiempo transcurrido: ${(Date.now() - startTime) / 1000} s.`;
                this.LOGGER.debug(
                  trace
                );
                await this.applicationLogService.updateState(applicationLogId, 8, `${trace}. Trabajando...`);
                await this.tenderService.upsert({
                  code: tenderDetails.CodigoExterno,
                  name: tenderDetails.Nombre,
                  description: tenderDetails.Descripcion,
                  details: tenderDetails,
                  isFavorite: false,
                });
              }
            }
          } catch (error) {
            const duration = Date.now() - startTime;
            await this.applicationLogService.updateState(applicationLogId, 9, `Duración: ${duration/1000} s. Error: ${inspect(error)}.`);
            this.LOGGER.error(
              `Error processing tenders for code ${tender.CodigoExterno} processed ${processed}/${tenders.Cantidad}`
            );
            continue;
          }
        }
        await this.applicationLogService.updateState(applicationLogId, 10, `processed ${processed}/${tenders.Cantidad} tiempo transcurrido: ${(Date.now() - startTime) / 1000} s.`);
      } else {
        this.LOGGER.debug(
          `No tenders found for args: ${JSON.stringify(args)}`
        );
      }
    } catch (error) {
      const duration = Date.now() - startTime;
      this.applicationLogService.updateState(applicationLogId, 9, `Duración: ${duration/1000} s. Error: ${inspect(error)}.`);
      this.LOGGER.error(
        `Error processing tenders for args: ${JSON.stringify(args)}`
      );
      return false;
    }
    this.LOGGER.debug('Finished processing tenders');
    return true;
    }
    async fetchTendersOfTheDay(): Promise<boolean | undefined> {
        if (!this.enabled) {
            this.LOGGER.log('Tender fetcher is disabled, skipping daily fetch.');
            return;
        }
        const now = moment().tz(TenderFetcherServiceImpl.TIMEZONE);
        const today = now.format('DDMMYYYY');
        const todayInWords = now.format('dddd, MMMM Do YYYY');
        this.LOGGER.debug(
        `Fetching tenders of the day ${todayInWords} (${TenderFetcherServiceImpl.TIMEZONE})...`
        );

        return await this.processTenders({ fecha: today });
    }
    async syncTenders(): Promise<void> {
        if (!this.enabled) {
            return;
        }
        this.LOGGER.debug(`Fetching tenders of the last ${this.fetchFromDaysAgo} days...`);
        const today = moment().tz(TenderFetcherServiceImpl.TIMEZONE);
        for (let i = 0; i < this.fetchFromDaysAgo; i++) {
        const date = today.subtract(i, 'days').format('DDMMYYYY');
        await this.processTenders({ fecha: date });
        }
    }
}
