import { HttpService } from '@nestjs/axios';
import { Inject, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { AxiosResponse } from 'axios';
import { firstValueFrom } from 'rxjs';
import { AdministrativeDivision, FirebaseService, GeoService, InsertNotificationRecord, PrimeLogger, User, UserRequestService, UserService } from 'src/framework';
import { NotificationRecordService } from 'src/framework/application/service/notification-record-service/notification-record-service.interface';
import { CompanyService, KeywordService, TenderRepository, UserCompanyService } from 'src/licitaapp/application';
import { ApplicationLogService } from 'src/licitaapp/application/service/application-log-service/application-log-service.interface';
import { HistoryTenderService } from 'src/licitaapp/application/service/history-tender-service/history-tender-service.interface';
import { MatchWordsService } from 'src/licitaapp/application/service/match-words-service/match-words-service.interface';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { UserCompanyTenderService } from 'src/licitaapp/application/service/user-company-tender-service/user-company-tender-service.interface';
import {
  Tender,
  DashboardTO,
  ParamsNotificationTO,
  DashboardDetailTO,
  RequestTender,
  Metadata,
  Word,
  CoarseTopics,
  Topics,
  EntityInfo,
  Licitacion,
  HistoryTender,
  InsertTender,
  HistoryTenderDetail,
} from 'src/licitaapp/domain';
import { CompanyUserTO } from 'src/licitaapp/domain/dto/company.user.to';
import { ApplicationTypeEnum } from 'src/licitaapp/domain/enum/enum.definition';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { inspect } from 'util';

@Injectable()
export class TenderServiceImpl implements TenderService {
  private readonly LOGGER = new PrimeLogger(TenderServiceImpl.name);
  private readonly maxTextRetries;
  private readonly textApiKey;
  private readonly textApiUrl;
  private readonly tenderLimitDays;
  private readonly isCronActive: boolean;
  constructor(
    @Inject('TenderRepository')
    private readonly tenderRepository: TenderRepository,
    @Inject('CompanyService') private readonly companyService: CompanyService,
    @Inject('HistoryTenderService')
    private readonly userHistoryTenderService: HistoryTenderService,
    @Inject('MatchWordsService')
    private readonly matchWordsService: MatchWordsService,
    @Inject('UserCompanyTenderService')
    private readonly userCompanyTenderRepository: UserCompanyTenderService,
    @Inject('NotificationRecordService')
    private readonly notificationRecordService: NotificationRecordService,
    @Inject('GeoService')
    private readonly geoService: GeoService,
    private readonly configService: ConfigService,
    private readonly httpService: HttpService,
    @Inject('FirebaseService')
    private readonly firebaseService: FirebaseService,
    @Inject('KeywordService')
    private readonly keywordService: KeywordService,
    @Inject('UserService')
    private readonly userService: UserService,
    @Inject('ApplicationLogService')
    private readonly applicationLogService: ApplicationLogService,
    @Inject('UserCompanyService')
    private readonly userCompanyService: UserCompanyService,
    @Inject('UserRequestService')
    private readonly userRequestService: UserRequestService,
  ) {
    this.textApiKey = this.configService.get<string>(
      'TEXTRAZOR_API_KEY',
    );
    this.textApiUrl = this.configService.get<string>(
      'TEXTRAZOR_API_URL',
    );
    this.maxTextRetries = this.configService.get<number>(
      'TEXTRAZOR_MAX_RETRIES',
      5
    );
    this.tenderLimitDays = this.configService.get<string>(
      'TENDERS_LIMIT_DAYS',
      '5'
    );
    this.isCronActive = this.configService.get<string>(
      'CRON_JOB_ENABLED',
      'true'
    ).toLowerCase() ===
    'true';
  }
  async erraseOldTenders(): Promise<string> {
    this.LOGGER.warn(`erraseOldTenders - START`);
    const idsTendersLogicalRemove = await this.tenderRepository.getLogicalRemoveTenderIds();
    this.LOGGER.warn(`erraseOldTenders - idsTendersLogicalRemove size: ${idsTendersLogicalRemove.length}`);

    const chunkSize = 500;
    for (let i = 0; i < idsTendersLogicalRemove.length; i += chunkSize) {
      const chunk = idsTendersLogicalRemove.slice(i, i + chunkSize);
      await this.userRequestService.erraseUserRequests(chunk);
      for(const tenderInfo of chunk){
        await this.userHistoryTenderService.erraseUserHistoryTender(tenderInfo);
      }
      await this.userCompanyTenderRepository.erraseUserCompanyTender(chunk);
      await this.notificationRecordService.erraseNotificationRecordsByTender(chunk.map(t => t.tenderId));
      await this.tenderRepository.erraseByListId(chunk.map(t => t.tenderId));
    }

    return 'true';
  }
  async tenderTaskPerDay(): Promise<void> {
    this.LOGGER.warn(`tenderTaskPerDay - START`);
    const applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.TENDER_DELETE_CLOSE_DATE_LOG);
    await this.logAndExecute(
      ApplicationTypeEnum.TENDER_DELETE_CLOSE_DATE_LOG,
      applicationLogId,
      this.checkTenderWithCloseDate.bind(this)
    );
  }
  
  async getListTenderHistory(userId: number): Promise<HistoryTender[]> {
    this.LOGGER.log(`LIST HISTORY_TENDER - userId: ${userId}`);
    return await this.userHistoryTenderService.getListTenderHistory(userId);
  }
  async logicalRemoveCompanyTenderUser(userId: number, companyId: number, tenderId: number): Promise<void> {
    this.LOGGER.log(`logicalRemoveCompanyTenderUser userId ${userId} companyId ${companyId} tenderId ${tenderId}`);
    return await this.userCompanyTenderRepository.logicalRemoveCompanyTenderUser(userId, companyId, tenderId);
  }
  async updateFavoriteUserCompanytender(userId: number, companyId: number, tenderId: number, isFavorite: boolean): Promise<void> {
    this.LOGGER.log(`updateFavoriteUserCompanytender - userId: ${userId}, companyId: ${companyId}, tenderId: ${tenderId}, isFavorite: ${isFavorite}`);
    return await this.userCompanyTenderRepository.updateFavoriteUserCompanyTender(userId, companyId, tenderId, isFavorite);
  }
  async upsert(tender: InsertTender): Promise<number> {
    this.LOGGER.log(`upsert tender - code: ${tender.code}`);
    return await this.tenderRepository.upsert(tender);
  }
  
  async tenderTaskByHour() {
    this.LOGGER.warn(`checkTenderByHour - START`);
    
    this.reviewTendersInfo(ApplicationTypeEnum.RECALCULATE_TENDER_DASHBOARD_LOG);
    this.reviewTendersInfo(ApplicationTypeEnum.CHECK_TENDER_FAVORITES_TO_CLOSE_LOG);
    this.reviewTendersInfo(ApplicationTypeEnum.DAILY_EMAIL_SUMMARY_LOG);
    
  }

  private async logAndExecute(type: ApplicationTypeEnum, applicationLogId:number, action: (applicationLogId:number) => Promise<string>) {
    this.LOGGER.log(`logAndExecute - type: ${type}, applicationLogId: ${applicationLogId}`);
    const startTime = Date.now();
    try {
      const message = await action(applicationLogId);
      const duration = Date.now() - startTime;
      await this.applicationLogService.updateState(applicationLogId, 10, `Duración: ${duration/1000} s. ${message}.`);
    } catch (error) {
      const duration = Date.now() - startTime;
      this.LOGGER.error(`logAndExecute - ${type} - Error: ${error}`);
      await this.applicationLogService.updateState(applicationLogId, 9, this.evalTextLarge(`Duración: ${duration/1000} s. Error: ${inspect(error)}.`));
    }
  }

  private evalTextLarge(text: string): string {
    if (text.length > 65535) {
      return text.substring(0, 65534);
    }
    return text;
  }
  

  async checkTenderFavoritesToClose(applicationLogId: number) {
    this.LOGGER.log(`checkTenderFavoritesToClose - START`);
    const arrayDaysToClose = this.tenderLimitDays.split('-');
    const messageTxt = `checkTenderFavoritesToClose - applicationLogId: ${applicationLogId} - arrayDaysToClose: ${arrayDaysToClose}`;
    this.LOGGER.log(`checkTenderFavoritesToClose - arrayDaysToClose: ${arrayDaysToClose}`);

    const usersIds = await this.userCompanyTenderRepository.findAllUserIdWithTenders();

    let daterequest:Date;
    for(const dateStr of arrayDaysToClose){
      daterequest = TenderUtil.subtractDays(TenderUtil.getSystemDateDDMMYYYY(), (+dateStr)*-1);
      messageTxt.concat(`. Processing date: ${daterequest.toLocaleDateString('es-CL', { day: '2-digit', month: '2-digit', year: 'numeric' })}`); 
      for(const userId of usersIds){
        const tenders = await this.userCompanyTenderRepository.getHistoryTendersFavorite(userId, daterequest);
        messageTxt.concat(`. Processing userId: ${userId} - tenders size: ${tenders.length}`);
        for(const tender of tenders){
          const notificationsArray: InsertNotificationRecord[] = [];
          notificationsArray.push({
            userId: userId,
            title: `Licitación favorita por cerrar`,
            defaultMessage: `Licitación ${tender.code} favorita por cerrar en ${dateStr} día(s)`,
            active: true,
            tenderId: tender.id,
            amountGeo: tenders.length,
          });
          //TODO: CANDIDATO A MEJORAR
          notificationsArray.length > 0 && await this.notificationRecordService.saveAll(notificationsArray);
        }
        await this.firebaseService.groupNotifications(userId);
      }
    }
    this.LOGGER.log(`checkTenderFavoritesToClose - END`);
    return messageTxt;
  }
  
  async checkNewTendersUserCompany(applicationLogId: number) {
    let systemHour = TenderUtil.getCurrentSystemDate();
    let messageTxt = `ApplicationLogId: ${applicationLogId} - START - systemHour: ${systemHour.getHours()}`;
    this.LOGGER.warn(`checkPerQuarterActions - applicationLogId: ${applicationLogId} - START - systemHour: ${systemHour.getHours()}`);
    if (systemHour.getHours() >= 8 && systemHour.getHours() <= 20) {
      this.LOGGER.warn(`Eval checkNewTenders8AM`);
      const usersIds = await this.userCompanyService.getUserIdsWithActiveCompany();
      this.LOGGER.warn(`checkPerQuarterActions - usersIds: ${usersIds.length}`);
      messageTxt = messageTxt.concat(`. Checking tender to ${usersIds.length} users`);
      for (const userId of usersIds) {
        const conpanyList = await this.userCompanyService.findCompaniesByUserId(userId);
        messageTxt = messageTxt.concat(`. Checking to userId: ${userId} with ${conpanyList.length} companies`);
        try {
          for(const companyItem of conpanyList){
            this.LOGGER.log(`checkPerQuarterActions - userId: ${userId} companyId: ${companyItem.id}`);
            await this.searchTendersDB({companyId: companyItem.id, userId}, false);
          }
          await new Promise(resolve => setTimeout(resolve, 2000));
        } catch (error) {
          this.LOGGER.error(`checkPerQuarterActions - reviewTenderState user: ${userId} - Error: ${error}`);
          throw error;
        }
      }
    }
    return messageTxt;
  }

  async tenderTaskBy2Hour(){
    this.LOGGER.warn(`checkTenderBy2Hour - START`);
    this.reviewTendersInfo(ApplicationTypeEnum.TENDER_WITHOUT_META_LOG);
    this.reviewTendersInfo(ApplicationTypeEnum.CHECK_NEW_TENDERS_USER_LOG);
  }

  private async createApplicationLogID( type: ApplicationTypeEnum) {
    const applicationLogId = await this.applicationLogService.save({
      userName: 'SYSTEM',
      statusTypeId: 8,
      detail: 'Iniciado a las ' + TenderUtil.getCurrentSystemDate().toLocaleTimeString(),
      type,
    });
    return applicationLogId;
  }

  async tenderTaskEvery5Minutes() {

    this.LOGGER.warn(`tenderTaskPer5Minutes - START`);

    this.reviewTendersInfo(ApplicationTypeEnum.TENDER_CHECK_CLOSE_DATE_LOG);
    this.reviewTendersInfo(ApplicationTypeEnum.TENDER_COMPLETE_STATE_LOG);
    this.reviewTendersInfo(ApplicationTypeEnum.TENDER_CHECK_SUBDIVISION_LOG);
    this.reviewTendersInfo(ApplicationTypeEnum.SUMMARY_ADMIN_LOG);
    
    this.LOGGER.warn(`tenderTaskPer5Minutes - END`);
  }

  async checkTenderSubdivision(applicationLogId: number) {
    const tendersWihoutCity = await this.tenderRepository.findWithoutSubdivision();
    var messageTxt = `ApplicationLogId: ${applicationLogId} - Licitaciones sin ciudades: ${tendersWihoutCity.length}`;
    if(tendersWihoutCity.length === 0){  
      this.LOGGER.log(`checkSubdivisionTender - No tenders without city`);
      return messageTxt;
    }

    const listCities = await this.geoService.findAdmdivisionsByCountry('CL');
    const listSubdivisionDB = await this.geoService.findSubdivisionsByCountryCode('CL');
    this.LOGGER.log(`checkSubdivisionTender - listCities size: ${listCities.length}`);
    for (const tenderDetails of tendersWihoutCity) {
      const detail = tenderDetails.details;
      detail.JustificacionPublicidad='';
      messageTxt = messageTxt.concat(`. Procesando licitación: ${tenderDetails.id} - ${tenderDetails.code}`);
      this.tenderRepository.updateSubdivision({
        id: tenderDetails.id,
        subdivisionId: await this.findSubdivision(listCities, detail, listSubdivisionDB)
      });
    }
    messageTxt = messageTxt.concat(`. Busqueda de ciudades en licitaciones finalizada.`);
    return messageTxt;
  }

  private isPrime(num: number): boolean {
    if (num < 2) return false;
    for (let i = 2; i <= Math.sqrt(num); i++) {
      if (num % i === 0) return false;
    }
    return true;
  }
  
  async checkCloseDate(applicationLogId: number) {
    const initTime = TenderUtil.getCurrentSystemDate().getTime();
    this.LOGGER.warn(`checkCloseDate - START`);
    const tendersWihoutCloseDate = await this.tenderRepository.findWithoutCloseDate();
    let messageTxt = `ApplicationLogId: ${applicationLogId} - Cantidad de Licitaciones sin fecha (Cierre) informada: ${tendersWihoutCloseDate.length}`;
    if(tendersWihoutCloseDate.length === 0){  
      this.LOGGER.log(`checkCloseDate - No tenders without city`);
      return messageTxt;
    }
    this.LOGGER.log(`checkCloseDate - listTenders size: ${tendersWihoutCloseDate.length}`);
    for (const tenderDetails of tendersWihoutCloseDate) {
      messageTxt = messageTxt.concat(`. Procesando licitación: ${tenderDetails.id} - ${tenderDetails.code}`);
      const detail = tenderDetails.details;
      detail.JustificacionPublicidad='';
      this.tenderRepository.updateCloseDate({
        id: tenderDetails.id,
        closeDate: detail.Fechas.FechaCierre?new Date(detail.Fechas.FechaCierre): null
      });
    }
    this.LOGGER.warn(`checkCloseDate - END - time ${TenderUtil.getCurrentSystemDate().getTime() - initTime} ms`);
    return messageTxt;
  }

  async checkMetadataKeywords(applicationLogId: number) {
    const initTime = TenderUtil.getCurrentSystemDate().getTime();
    const listKeywordsToUpdate = await this.keywordService.getAllWithouthMetadata();
    var messageTxt = `ApplicationLogId: ${applicationLogId} - Tamaño lista a actualizar Keywords: ${listKeywordsToUpdate.length}`;
    this.LOGGER.debug(`checkMetadataKeywords - listKeywordsToUpdate size: ${listKeywordsToUpdate.length}`);

    if(listKeywordsToUpdate.length === 0){
      this.LOGGER.debug(`checkMetadataKeywords - No keywords without metadata`);
      return messageTxt;
    }

    for(const keyword of listKeywordsToUpdate){
      this.LOGGER.debug(`checkMetadataKeywords - keyword: ${keyword.value}`);
      try{
        const metadata = await this.fetchRazorMetadata(keyword.value);
        await this.keywordService.updateMetadata(keyword.id, this.createResponseTenderMetadata(metadata));
        messageTxt = messageTxt.concat(`. Keyword ID: ${keyword.id} actualizado con éxito`);
      }catch(error){
        this.LOGGER.error(`checkMetadataKeywords - Metadata Keyword id ${keyword.id} error: ${error}`);
        //TODO: arreglar el error y controlarlo de otra manera.
        throw {
          message: `Fallo al actualizar metadata de la keyword: ${keyword.id} - ${keyword.value}`,
          error,
        };
      }
    }
    this.LOGGER.debug(`checkMetadataKeywords - time ${TenderUtil.getCurrentSystemDate().getTime() - initTime} ms`);
    return messageTxt;
  }

  async checkTenderWithCloseDate(applicationLogId: number) {
    const { end } = TenderUtil.getDayStartAndEnd(TenderUtil.getCurrentSystemDate());
    const listTenderToDelete =await this.tenderRepository.findByCloseDate(TenderUtil.subtractDays(end, 1));
    var messageTxt = `ApplicationLogId: ${applicationLogId} - Cantidad de Licitaciones vencidas el día de ayer: ${listTenderToDelete.length}`;
    for (const tenderDetails of listTenderToDelete) {
      messageTxt = messageTxt.concat(`. Procesando licitación: ${tenderDetails.id} - ${tenderDetails.code}`);
      this.LOGGER.log(`logicalRemoveByCloseDate - tender to delete: ${tenderDetails.id}`);
      this.tenderRepository.logicalRemove(tenderDetails.id);
    }
    return messageTxt;
  }

  async tenderTaskPerMinutes() {
    const usage = process.memoryUsage();
    this.LOGGER.debug(`MEMORY IN USE: RSS=${usage.rss / 1024 / 1024} MB, HEAP=${usage.heapUsed / 1024 / 1024} MB`);
    
    this.reviewTendersInfo(ApplicationTypeEnum.KEYWORDS_LOG);
  }

  async createDashboardCompany(companyId: number, userId: number) {
    await this.searchTenderToUserCompanyWithSubdivision(companyId, userId);
    this.searchTendersDB({companyId, userId}, true);
    await this.firebaseService.groupNotifications(userId);    
  }

  async searchTenderToUserCompanyWithSubdivision(companyId: number, userId: number) {

    const listSubdivisionByCompany = await this.companyService.getSubdivisionsByCompanyId(companyId);
    const notificationsArray: InsertNotificationRecord[] = [];
    for(const subdivision of listSubdivisionByCompany){
      this.LOGGER.log(`searchTenderToUserCompany - subdivision: ${subdivision.name} to user: ${userId} company: ${companyId}`);
      const listTenderBySubdivision = await this.tenderRepository.findBySubdivisionId(subdivision.id, true);
      this.LOGGER.log(`searchTenderToUserCompany - subdivision: ${subdivision.id} 
        tenders: ${listTenderBySubdivision?listTenderBySubdivision.length:0}`);
      if(listTenderBySubdivision && listTenderBySubdivision.length > 0){
        this.userCompanyTenderRepository.saveAll(userId, companyId, 
          listTenderBySubdivision.map(tender => tender.id), `Cerca en ${subdivision.name}`);
        
          notificationsArray.push({
            userId: userId,
            title: `Encontradas por Ubicación`,
            defaultMessage: (`Encontradas ${listTenderBySubdivision.length} licitaciones en ${subdivision.name}`.length > 255) ? 
              `Encontradas ${listTenderBySubdivision.length} licitaciones en ${subdivision.name}`.substring(0, 255) : 
              `Encontradas ${listTenderBySubdivision.length} licitaciones en ${subdivision.name}`,
            active: true,
            subdivisionId: subdivision.id,
            amountGeo: listTenderBySubdivision.length,
          });
          this.LOGGER.warn(`Save success to user-company-tender user: ${userId} company: ${companyId} subdivision ${subdivision.id}`);
      }
    }
    notificationsArray.length > 0 && await this.notificationRecordService.saveAll(notificationsArray); 
  }

  async reviewTenderWihoutMetadata(applicationLogId: number) {
    const listTenderMetadataToUpdate = await this.tenderRepository.getAllWithouthMetadata();
    this.LOGGER.log(`ReviewTendersInfo - listTenderMetadataToUpdate size: ${listTenderMetadataToUpdate.length}`);
    var messageTxt = `ApplicationLogId: ${applicationLogId} - Licitaciones sin metadatos: ${listTenderMetadataToUpdate.length}`;
    let count = listTenderMetadataToUpdate.length;
    for(const tender of listTenderMetadataToUpdate){
      messageTxt = messageTxt.concat(`. Procesando licitación: ${tender.id}`);
      this.LOGGER.log(`ReviewTendersInfo - tender: ${tender.id}`);
      count--;
      try{
        const metadata = await this.fetchRazorMetadata(this.createRequestTextTender(tender.description, tender.detail));
        await this.tenderRepository.updateMetadata(tender.id, this.createResponseTenderMetadata(metadata));
      }catch(error){
        if((error.response.data.error as string).includes('You have reached your daily TextRazor request limit.')){
          messageTxt = messageTxt.concat(`. Limit_reached - ${count} tenders left to process`);
          return messageTxt;
        }else {
          messageTxt = messageTxt.concat(`. Error al procesar licitación: ${tender.id} - ${error.message}`);
        }
        this.LOGGER.error(`ReviewTendersInfo - Metadata Tender id ${tender.id} error: ${inspect(error)}`);
      }
    }
    return messageTxt;
  }

  async reviewTenderStatus(applicationLogId: number) {
    const { start, end } = TenderUtil.getDayStartAndEnd(new Date());
    this.LOGGER.log(`ReviewTendersInfo - start: ${start} end: ${end}`);
    const tenders = await this.tenderRepository.findByDatesUpset(start, end);
    this.LOGGER.log(`ReviewTendersInfo - tenders size: ${tenders.length}`);
    var messageTxt = `ApplicationLogId: ${applicationLogId} - Licitaciones a revisar si ha cambiado su estado: ${tenders.length}`;

    for (const tenderDetails of tenders) {
      const detail = tenderDetails.details;
      const isActive = this.evalStateTender(detail);
      this.tenderRepository.updateLogicalRemove({
        id: tenderDetails.id,
        state: isActive
      });
    }
    messageTxt = messageTxt.concat(`. Revisión de estado de licitaciones finalizada.`);
    return messageTxt;
  }

  /**
   * 1. rescatamos las licitaciones actualizadas o creadas el dia de hoy
   */
  async reviewTendersInfo(action: ApplicationTypeEnum): Promise<void> {
    this.LOGGER.warn(`ReviewTendersInfo - START - action: ${action}`);
    const initTime = TenderUtil.getCurrentSystemDate().getTime();
    this.LOGGER.warn(`ReviewTendersInfo - parcial time ${TenderUtil.getCurrentSystemDate().getTime() - initTime} ms`);
    if(action === ApplicationTypeEnum.TENDER_WITHOUT_META_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.TENDER_WITHOUT_META_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.TENDER_WITHOUT_META_LOG,
        applicationLogId,
        this.reviewTenderWihoutMetadata.bind(this)
      );
    }else if(action === ApplicationTypeEnum.TENDER_COMPLETE_STATE_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.TENDER_COMPLETE_STATE_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.TENDER_COMPLETE_STATE_LOG,
        applicationLogId,
        this.reviewTenderStatus.bind(this)
      );
    }else if(action === ApplicationTypeEnum.TENDER_CHECK_CLOSE_DATE_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.TENDER_CHECK_CLOSE_DATE_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.TENDER_CHECK_CLOSE_DATE_LOG,
        applicationLogId,
        this.checkCloseDate.bind(this)
      );
    }else if(action === ApplicationTypeEnum.TENDER_CHECK_SUBDIVISION_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.TENDER_CHECK_SUBDIVISION_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.TENDER_CHECK_SUBDIVISION_LOG,
        applicationLogId,
        this.checkTenderSubdivision.bind(this)
      );
    }else if(action === ApplicationTypeEnum.TENDER_DELETE_CLOSE_DATE_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.TENDER_DELETE_CLOSE_DATE_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.TENDER_DELETE_CLOSE_DATE_LOG,
        applicationLogId,
        this.checkTenderWithCloseDate.bind(this)
      );
    }else if(action === ApplicationTypeEnum.SUMMARY_ADMIN_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.SUMMARY_ADMIN_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.SUMMARY_ADMIN_LOG,
        applicationLogId,
        this.sendEmailSumaryToAdmin.bind(this)
      );
    }else if(action === ApplicationTypeEnum.KEYWORDS_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.KEYWORDS_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.KEYWORDS_LOG,
        applicationLogId,
        this.checkMetadataKeywords.bind(this)
      );
    }else if(action === ApplicationTypeEnum.CHECK_TENDER_FAVORITES_TO_CLOSE_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.CHECK_TENDER_FAVORITES_TO_CLOSE_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.CHECK_TENDER_FAVORITES_TO_CLOSE_LOG,
        applicationLogId,
        this.checkTenderFavoritesToClose.bind(this)
      );
    }else if(action === ApplicationTypeEnum.DAILY_EMAIL_SUMMARY_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.DAILY_EMAIL_SUMMARY_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.DAILY_EMAIL_SUMMARY_LOG,
        applicationLogId,
        this.firebaseService.sendDailyEmailSumary.bind(this)
      );
    }else if(action === ApplicationTypeEnum.CHECK_NEW_TENDERS_USER_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.CHECK_NEW_TENDERS_USER_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.CHECK_NEW_TENDERS_USER_LOG,
        applicationLogId,
        this.checkNewTendersUserCompany.bind(this)
      );
    }else if(action === ApplicationTypeEnum.RECALCULATE_TENDER_DASHBOARD_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.RECALCULATE_TENDER_DASHBOARD_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.RECALCULATE_TENDER_DASHBOARD_LOG,
        applicationLogId,
        this.recalculteTender.bind(this)
      );
    }else if(action === ApplicationTypeEnum.ERRASE_OLD_TENDERS_LOG){
      let applicationLogId = await this.createApplicationLogID(ApplicationTypeEnum.ERRASE_OLD_TENDERS_LOG);
      await this.logAndExecute(
        ApplicationTypeEnum.ERRASE_OLD_TENDERS_LOG,
        applicationLogId,
        this.erraseOldTenders.bind(this)
      );
    }
    this.LOGGER.warn(`ReviewTendersInfo - tenders updated END - time ${TenderUtil.getCurrentSystemDate().getTime() - initTime} ms`);
  }

  private createResponseTenderMetadata(metadata: any) {
    let responseMetadata = new Metadata();
    this.LOGGER.log(`createResponseTenderMetadata - EVAL SENTENCES WORDS`);
    if(metadata.response.sentences){
      responseMetadata.words = [];
      for(const sentence of metadata.response.sentences){
        for(const word of sentence.words){
          responseMetadata.words.push(
            new Word(
              word.position,
              word.startingPos,
              word.endingPos,
              word.stem,
              word.token,
              word.partOfSpeech
            )
          );
        }
      }
    }
    this.LOGGER.log(`createResponseTenderMetadata - EVAL COURSE TOPISC`);
    if(metadata.response.coarseTopics){
      responseMetadata.coarseTopics = [];
      for(const coarseTopic of metadata.response.coarseTopics){
        responseMetadata.coarseTopics.push(
          new CoarseTopics(coarseTopic.label, coarseTopic.score));
      }
    }
    this.LOGGER.log(`createResponseTenderMetadata - EVAL TOPICS`);
    if(metadata.response.topics){
      responseMetadata.topics = [];
      for(const topic of metadata.response.topics){
        responseMetadata.topics.push(new Topics(topic.label, topic.score));
      }
    }
    this.LOGGER.log(`createResponseTenderMetadata - EVAL ENTITIES`);
    if(metadata.response.entities){
      responseMetadata.entities = [];
      for(const entity of metadata.response.entities){
        responseMetadata.entities.push(
          new EntityInfo(entity.entityId, entity.type, entity.confidenceScore));
      }
    }    
    return responseMetadata;
  }

  async sendEmailSumaryToAdmin(applicationLogId:number){
    this.LOGGER.log(`sendEmailSumaryToAdmin - START`);
    return await this.firebaseService.sendEmailSumaryToAdmin(applicationLogId, this.userService);
  }

  private createRequestTextTender(description: string, detail: Licitacion) {
    let out = description;
    for(const itemDetail of detail.Items.Listado){
      if(itemDetail.Descripcion){
        out = out + ' ' + itemDetail.Descripcion;
      }
      if(itemDetail.Categoria){
        out = out + ' ' + itemDetail.Categoria;
      }
    }
    return out;
  }

  evalStateTender(detail: Licitacion) {
    return detail.Estado === 'Publicada' || detail.Estado === 'Desierta' || detail.Estado === 'Adjudicada';
  }

  private async findSubdivision(listCities: AdministrativeDivision[], detail: Licitacion, listSubdivisionDB: AdministrativeDivision[]) {
    let subdivisionId = null;
    let searchCity = detail.Comprador && detail.Comprador.ComunaUnidad;
       
    for (const city of listCities) {
      if (searchCity && this.normalizeString(searchCity.trim()) === this.normalizeString(city.name.trim())) {
        subdivisionId = city.subidivisionId;
        break;
      }
    }

    if(subdivisionId === null){
      searchCity = detail.Comprador && detail.Comprador.RegionUnidad;
      for (const subdivision of listSubdivisionDB) {
        if (searchCity && this.normalizeString(searchCity.trim()) === this.normalizeString(subdivision.name.trim())) {
          subdivisionId = subdivision.id;
          break;
        }
      }
    }      
    this.LOGGER.log(`findCity - subdivisionId: ${subdivisionId} - city: ${searchCity}`);
    return subdivisionId;
  }

  private normalizeString(str: string): string {
    return str
      .normalize("NFD") // Descompone caracteres con tilde
      .replace(/[\u0300-\u036f]/g, "") // Elimina los diacríticos (tildes)
      .toLowerCase(); // Convierte todo a minúsculas
  }
  
  async getTenderByCode(code: string, userId?: number) {
    const tender = await this.tenderRepository.findByCode(code);
    if (tender) {
      tender.labelAmount = TenderUtil.convertToMillions(tender.details.MontoEstimado || 0);
      tender.typeOfMoney = tender.details.Moneda;
      tender.labelLastUpdated = TenderUtil.calculateTime(tender.createdAt, tender.updatedAt);
      if(userId){
        this.LOGGER.log(`SAVE INTO HISTORY_TENDER - code: ${code}, userId: ${userId} isFavorite: false`);
        this.userHistoryTenderService.saveHistoryTender(userId, tender.id);
      }
    }
    return tender;
  }

  async addTenderByCodeToCompany(code: string, companyId: number): Promise<string> {
    this.LOGGER.log(`addTenderByCodeToCompany - code: ${code}, companyId: ${companyId}`);
    const tenderMatch = await this.tenderRepository.findByCode(code);
    if (!tenderMatch) {
      this.LOGGER.warn(`addTenderByCodeToCompany - Tender not found for code: ${code}`);
      return 'Licitación no encontrada';
    }
    const company = await this.companyService.findById(companyId);
    if (!company) {
      this.LOGGER.warn(`addTenderByCodeToCompany - Company not found for companyId: ${companyId}`);
      return 'Empresa no encontrada';
    } 
    const usersCompany = await this.userCompanyTenderRepository.getUsersIdsByCompanyId(companyId);
    const notificationsArray: InsertNotificationRecord[] = [];
    for (const userId of usersCompany) {
      this.LOGGER.log(`addTenderByCodeToCompany - userId: ${userId} - code: ${code}`);
          this.userCompanyTenderRepository.save(userId, companyId, tenderMatch.id, 
            `Licitación agregada por tu administrador.`, '');
            notificationsArray.push({
              userId: userId,
              title: `Coincidencias por palabras claves`,
              defaultMessage: `Creemos que esta licitación puede ser de tu interés`,
              active: true,
              monthFilterTender: null,
              amountWords: 0,
              amountGeo: 0
            });
    }

    notificationsArray.length > 0 && await this.notificationRecordService.saveAll(notificationsArray); 

    return 'estado';
  }

  async generateInfoDashboard(user: User, companyId: number): Promise<DashboardTO> {
    let output = new DashboardTO();

    let listDetail: DashboardDetailTO[] = [];
    let amount = 0;

    const company = await this.companyService.findById(companyId);
    let amountFavorites = 0;
    let countFavorites = 0;
    if (company) {
      const tenderListByCompany = await this.userCompanyTenderRepository.findTenderByUserCompanyId(user.id, company.id);
      amount += tenderListByCompany.length;

      tenderListByCompany.forEach((tender) => {
        countFavorites = tender.isFavorite ? countFavorites + 1 : countFavorites;
        amountFavorites = tender.isFavorite ? amountFavorites + tender.details.MontoEstimado || 0 : amountFavorites;
        if(!tender.isFavorite){
          const existingDetail = listDetail.find(detail => detail.description === tender.source);
          if (existingDetail) {
            existingDetail.amountPerSubdivision += tender.details.MontoEstimado || 0;
            existingDetail.labelAmount = `${existingDetail.amountPerSubdivision} (${++existingDetail.count!} coincidencias)`;
          } else {
            listDetail.push({
              subdivisionId: tender.subdivisionId || null,
              description: tender.source!,
              amountPerSubdivision: tender.details.MontoEstimado || 0,
              labelAmount: `${tender.details.MontoEstimado || 0} (1 coincidencia)`,
              count: 1,
            });
          }          
        }
      });
      if(countFavorites > 0){
        listDetail.push({
          description: 'Mis favoritos',
          amountPerSubdivision: amountFavorites,
          labelAmount: `${countFavorites} ${countFavorites==1 ? 'en seguimiento' : 'en seguimiento'}`,
        });
      }
    }

    
    output.userAmount = await this.userCompanyService.countUserByCompany(companyId);
    output.tenderTotalAmount = await this.tenderRepository.countActiveTenders();

    output.newTenderAmount = (amount - countFavorites < 0) ? 0 : amount - countFavorites;
    output.listDetails = listDetail.map((detail) => {
      if(detail.description ==='Mis favoritos' ) {
        return detail;
      }
      return {
        description: detail.description,
        amountPerSubdivision: detail.amountPerSubdivision,
        subdivisionId: detail.subdivisionId,
        labelAmount: TenderUtil.convertToMillions(detail.amountPerSubdivision) + ` (${detail.count} ${detail.count==1 ? 'coincidencia' : 'coincidencias'})`,
        dateFilterTender: TenderUtil.evalMonth(detail.description),
      };
    });
    
    return output;
  }
  
  

  async getPaginatedTenders(userId: number, companyId: number, page: number, pageSize: number, searchType: string, subdivisionId: number, monthRequest: string): Promise<Tender[]> {
    this.LOGGER.log(`GET PAGINATED TENDERS - userId: ${userId}, companyId: ${companyId}, page: ${page}, pageSize: ${pageSize}, searchType: ${searchType}, subdivisionId: ${subdivisionId} monthRequest: ${monthRequest}`);
    return (await this.userCompanyTenderRepository.getPaginatedTenders(userId, companyId, page, pageSize, searchType, subdivisionId, monthRequest)).map(
      (tender) => {
        tender.labelAmount = TenderUtil.convertToMillions(tender.details.MontoEstimado || 0);
        tender.typeOfMoney = tender.labelAmount==='0' ? `${tender.details.Moneda} - ${tender.details.Tipo}`: tender.details.Moneda;
        tender.labelLastUpdated = TenderUtil.calculateTime(tender.createdAt, tender.updatedAt);
        const closeTender = TenderUtil.calculateTimeFromDateString(tender.details.Fechas.FechaCierre);
        const dateInfo = closeTender.length>0 ? `Fecha de cierre: ${tender.details.Fechas.FechaCierre?.substring(0, 16).replace('T', ' ')} ${closeTender === 'Finalizado' ? '': '\n ('+closeTender}) ` : 'Fecha de cierre no informada';
        const publishDate = TenderUtil.calculateTimeFromDateString(tender.details.Fechas.FechaPublicacion);
        const datePublishInfo = publishDate.length>0 ? `\n Fecha de publicación: ${tender.details.Fechas.FechaPublicacion?.substring(0, 16).replace('T', ' ')}` : 'Fecha de publicación no informada';
        tender.labelCloseTender =  dateInfo + datePublishInfo +' \n Estado: ' + tender.details.Estado;
        return tender;
      },
    );
  }

  async getPaginatedHistoryTenders(userId: number, page: number, pageSize: number): Promise<HistoryTenderDetail[]> {
    const resultPaginated = await this.userCompanyTenderRepository.getPaginatedHistoryTenders(userId, page, pageSize);
    const detailsTender = await this.tenderRepository.findInfohistoryTender(resultPaginated.map(tender => tender.id));
    this.LOGGER.log('size resultPaginated: '+resultPaginated.length + ' - detailsTender: '+detailsTender.length);
    return resultPaginated.map(
      (tender) => {
        if(!tender.isErrased){
          const detailTender = detailsTender.find(detail => detail.tenderId === tender.id);
          if(detailTender){
            tender.code = detailTender.code;
            tender.name = `${detailTender.code} - ${detailTender.details.Nombre.toUpperCase() || ''}`;
            tender.description = detailTender.details.Descripcion || '';
            tender.labelAmount = TenderUtil.convertToMillions(detailTender.details.MontoEstimado || 0);
            tender.typeOfMoney = tender.labelAmount==='0' ? `${detailTender.details.Moneda} - ${detailTender.details.Tipo}`: detailTender.details.Moneda;
            tender.labelLastUpdated = TenderUtil.calculateTime(detailTender.createdAt, detailTender.updatedAt);
            const closeTender = TenderUtil.calculateTimeFromDateString(detailTender.details.Fechas.FechaCierre);
            const dateInfo = closeTender.length>0 ? `Fecha de cierre: ${detailTender.details.Fechas.FechaCierre?.substring(0, 16).replace('T', ' ')} ${closeTender === 'Finalizado' ? '': '\n ('+closeTender}) ` : 'Fecha de cierre no informada';
            const publishDate = TenderUtil.calculateTimeFromDateString(detailTender.details.Fechas.FechaPublicacion);
            const datePublishInfo = publishDate.length>0 ? `\n Fecha de publicación: ${detailTender.details.Fechas.FechaPublicacion?.substring(0, 16).replace('T', ' ')}` : 'Fecha de publicación no informada';
            tender.labelCloseTender =  dateInfo + datePublishInfo +' \n Estado: ' + detailTender.details.Estado;
          }
        }
        return tender;
      },
    );
  }

  
  async searchTendersDB(params: ParamsNotificationTO, waitMinutes: boolean = false): Promise<String> {

    this.LOGGER.log(`SEARCH TENDERS waitMinutes: ${waitMinutes}`);
    if(waitMinutes){
      await new Promise(resolve => setTimeout(resolve, 1000 * 60 * 2));
    }
    this.LOGGER.log(`SEARCH TENDERS - START`);

    const keywordsCompany = await this.keywordService.findByCompanyId(params.companyId, true);

    const outListKeywordsCompany: string[] = [];
    keywordsCompany.forEach(keyword => {
      const { wordsList, topicsList, coarseTopicsList } = this.getListOfWordsMetada(keyword.metadata!);
      outListKeywordsCompany.push(...wordsList.map(word => word.stem));
      outListKeywordsCompany.push(...topicsList.map(topic => topic.label));
      outListKeywordsCompany.push(...coarseTopicsList.map(coarseTopic => coarseTopic.label));
    });

    if(keywordsCompany.length === 0){
      this.LOGGER.warn(`SEARCH TENDERS - No keywords found for company ${params.companyId}`);
      return 'false';
    }
    
    for(var month = 2; month >= 0; month--){
      const listMatchedTenders : RequestTender[] = [];
      const { start, end } = TenderUtil.getMonthStartAndEnd(new Date(), -month);
      const tenders = await this.tenderRepository.findByDates(start, end);
      this.LOGGER.log(`SEARCH TENDERS - tenders size: ${tenders.length} form Date ${start} to ${end}`);
      
      for(const tender of tenders){

        const { wordsList, topicsList, coarseTopicsList } = this.getListOfWordsMetada(tender.metadata!);

        const findMatchCoarseTopic = this.matchWordsService.findMatches(
          outListKeywordsCompany,
          coarseTopicsList.map(coarseTopic => coarseTopic.label),
        );

        const findMatchTopics = this.matchWordsService.findMatches(
          outListKeywordsCompany,
          topicsList.map(topic => topic.label),
        );

        const findMatchWords = this.matchWordsService.findMatches(
          outListKeywordsCompany,
          wordsList.map(word => word.stem),
        );

        const havematchList = this.evalResultsMatch(tender.id, findMatchTopics, findMatchWords, findMatchCoarseTopic);
        if(havematchList.length > 5){
          tender.matchResult = JSON.stringify({ keywordsOrigin: outListKeywordsCompany, havematchList });
          listMatchedTenders.push(tender);
        }

      }
      
      if(listMatchedTenders.length > 0){

        for(const tenderMatch of listMatchedTenders){
          this.LOGGER.log(`SAVING TENDERS ${tenderMatch.id} for user ${params.userId} and company ${params.companyId}`);
          this.userCompanyTenderRepository.save(params.userId, params.companyId, tenderMatch.id, 
            `Coincidencias por palabras claves ${start.toLocaleString('es-CL', { month: 'long' })} ${start.getFullYear()}`, tenderMatch.matchResult!);
        }       
    
        const notificationsArray: InsertNotificationRecord[] = [];
        notificationsArray.push({
          userId: params.userId,
          title: `Coincidencias por palabras claves`,
          defaultMessage: `Licitaciones encontrada en ${start.toLocaleString('es-CL', { month: 'long' })} ${start.getFullYear()}`,
          active: true,
          monthFilterTender: start,
          amountWords: listMatchedTenders.length,
          amountGeo: 0
        });
    
        notificationsArray.length > 0 && await this.notificationRecordService.saveAll(notificationsArray); 
      }
    }
    await this.firebaseService.groupNotifications(params.userId);
    this.LOGGER.warn(`SEARCH TENDERS - END`);
    return 'true';
  }

  private evalResultsMatch(tenderId:number, findMatchTopics: { item: string; matches: string[]; }[], 
    findMatchWords: { item: string; matches: string[]; }[], findMatchCoarseTopic: { item: string; matches: string[]; }[]) {
    
    const listWithMatch: string[] = [];
      
    findMatchTopics.forEach((result) => {
      if(result.matches.length > 0){
        listWithMatch.push(result.item.toLowerCase());
      }
    });

    /*findMatchWords.forEach((result) => {
      if(result.matches.length > 0){
        listWithMatch.push(result.item.toLowerCase());
      }
    }
    );

    findMatchCoarseTopic.forEach((result) => {
      if(result.matches.length > 0){
        listWithMatch.push(result.item.toLowerCase());
      }
    });*/

    return [...new Set(listWithMatch)];
  }

  // Sensibilidad de las palabras claves
  private getListOfWordsMetada(json: Metadata): { wordsList: Word[]; topicsList: Topics[]; coarseTopicsList: CoarseTopics[] } {
    let wordsList = [];
    if(json.words){
      wordsList = json.words.filter((word: Word) =>
        word.partOfSpeech === 'NOUN' || word.partOfSpeech === 'ADJ'
      );
    }
  
    let topicsList = [];
    if(json.topics){
      topicsList = json.topics.filter((topic: Topics) =>
        topic.score > 0.6
      );
    }
    let coarseTopicsList = [];
    if(json.coarseTopics){
      coarseTopicsList = json.coarseTopics.filter((coarseTopic: CoarseTopics) =>
        coarseTopic.score > 1
      );
    }
  
    return {
      wordsList,
      topicsList,
      coarseTopicsList
    };
  }
  

  async fetchRazorMetadata(textToEval: string) {
    this.LOGGER.log(`Fetching metadata for text: ${textToEval}`);
    const delay = (ms: number) =>
      new Promise((resolve) => setTimeout(resolve, ms));
  
    for (let attempt = 1; attempt <= this.maxTextRetries; attempt++) {
      try {
        
        const formData = new URLSearchParams();
        formData.append('text', textToEval);
        formData.append('extractors', 'entities,topics,entailments');
  
        const response: AxiosResponse = await firstValueFrom(
            this.httpService.post(this.textApiUrl!, formData, {
              headers: {
              'x-textrazor-key': this.textApiKey,
              'Content-Type': 'application/x-www-form-urlencoded',
            },
          }),
        );
        
        return response.data;
      } catch (error) {
        if (attempt === Number(this.maxTextRetries)) {
          this.LOGGER.debug(`Failed to fetch tender details for args: ${JSON.stringify(textToEval)}`);
          throw error;
        }
        this.LOGGER.debug(
          `Attempt ${attempt} failed. Retrying in ${attempt * 5}ms...`
        );
        await delay(attempt * 5); // Exponential backoff
      }
    }
  }

  async recalculteTender(applicationLogId:number){
    this.LOGGER.log('recalculateTender - START');
    const companyIds = await this.companyService.getCompanyIdsToRecalculateTender();
    this.LOGGER.log(`recalculateTender - companyIds size: ${companyIds.length}`);
    let messageTxt = `Recalculating tenders for ${companyIds.length} companies`;
    await this.applicationLogService.updateState(applicationLogId, 8, messageTxt);

    for(const companyId of companyIds){
      this.LOGGER.log(`recalculateTender - applicationLogId: ${applicationLogId} - errase actual tenders - companyId: ${companyId}`);
      messageTxt = messageTxt.concat('. Errase tender to companyId: ' + companyId);
      this.userCompanyTenderRepository.erraseJoinCompanyTender(companyId);
      this.LOGGER.log(`recalculateTender - applicationLogId: ${applicationLogId} - search new tenders dashboard - companyId: ${companyId}`);
      const activeUserIds = await this.userCompanyService.getUserIdsWithActiveCompany();
      messageTxt = messageTxt.concat('. Loading new tenders to companyId: ' + companyId + ' and users: ' + activeUserIds.length);
      for(const userId of activeUserIds){
        this.LOGGER.log(`recalculateTender - applicationLogId: ${applicationLogId} - create dashboard for userId: ${userId} companyId: ${companyId}`);
        this.createDashboardCompany(companyId, userId);
      }
      this.companyService.updateCheckTenders(companyId, true);
    } 
    return messageTxt;
  }

  async paginationByCompanyAdmin(
    page: number,
    pageSize: number,
  ): Promise<CompanyUserTO[]> {
    this.LOGGER.log(`Paginating companies by admin, page: ${page}, pageSize: ${pageSize}`);
    const companies = await this.companyService.paginationByCompanyAdmin(page, pageSize);
    for (const company of companies) {
      const userInfo = await this.userCompanyService.getInfoUserCompany(
        company.companyId,
      );
      company.listUsers = userInfo;
      company.amountTenders = await this.userCompanyTenderRepository.countActiveTendersCompany(company.companyId);
    }
    return companies;
  }

}

